/*
 * hxb2_mqtt.c
 *
 *  Created on: 2025. 11. 15.
 *      Author: c6h6
 */

// main/hxb2_mqtt.c

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "freertos/FreeRTOS.h"
#include "freertos/task.h"

#include "esp_event.h"
#include "esp_log.h"
#include "mqtt_client.h"

#include "dummy_telemetry.h" // 방금 생성한 헤더
#include "portmacro.h"

#include "driver/gpio.h"
#include "driver/uart.h"

#define HXB2_DEVICE_ID "HXB2k"
#define HXB2_MQTT_URI "mqtt://192.168.121.4:1883"

static const char *TAG = "HXB2MQTT";
extern char dev_uuid[5];
bool subscribed = false;
static esp_mqtt_client_handle_t s_mqtt = NULL;
static bool s_telemetry_task_started = false;
static bool s_lease_task_started = false;

esp_mqtt_client_handle_t g_mqtt_client = NULL;

/* --------------------------------------------------------------------------
 *  STM32와 연결된 UART 설정 (보드 맞게 핀만 바꿔 쓰면 됨)
 * -------------------------------------------------------------------------- */
#define STM32_UART_PORT UART_NUM_1
#define STM32_UART_TX_GPIO 17
#define STM32_UART_RX_GPIO 16

/* --------------------------------------------------------------------------
 *  JSON → 내부 Command 구조체 정의
 * -------------------------------------------------------------------------- */

typedef struct {
	// 메타
	uint32_t cmd_rev;
	bool has_cmd_rev;

	// Digital fields
	bool has_PSU48_ENA;
	bool PSU48_ENA;

	bool has_ELECTROLYTE_CTRL_ENA;
	bool ELECTROLYTE_CTRL_ENA;

	bool has_HIGHP_PRODUCTION_ENA;
	bool HIGHP_PRODUCTION_ENA;

	// Analog fields
	bool has_PSU48_V_SET;
	float PSU48_V_SET;

	bool has_PSU48_I_SET;
	float PSU48_I_SET;

	bool has_ELECTROLYTE_T_SP_DEGC;
	float ELECTROLYTE_T_SP_DEGC;

	bool has_ELECTROLYTE_FLOW_SP_SLM;
	float ELECTROLYTE_FLOW_SP_SLM;
} hxb_cmd_desired_t;

typedef struct {
	bool has_time;
	uint16_t year;	// full year, e.g. 2025
	uint8_t month;	// 1..12
	uint8_t day;	// 1..31
	uint8_t hour;	// 0..23
	uint8_t minute; // 0..59
	uint8_t second; // 0..59
} hxb_cmd_lease_t;

// 현재 HMI에서 내려온 "최신 Desired 상태"를 들고 있는 캐시
hxb_desired_shadow_t s_desired_shadow = {.cmd_rev_initialized = false,
										 .psu48_ena = false,
										 .electrolyte_ctrl_ena = false,
										 .highp_production_ena = false,
										 .psu48_v_set = 0.0f,
										 .psu48_i_set = 0.0f,
										 .electrolyte_t_sp_degC =
											 25.0f, // 초기값은 취향/사양에 맞게
										 .electrolyte_flow_sp_slm = 0.0f};
typedef struct {
	bool emergency; // true면 EMERGENCY_STOP
} hxb_cmd_emergency_t;

/* --------------------------------------------------------------------------
 *  간단 JSON 파서 유틸 (cJSON 없이 key 기반으로만 파싱)
 * -------------------------------------------------------------------------- */

static const char *find_key(const char *json, const char *key) {
	// "\"KEY\"" 패턴 찾기
	char pattern[64];
	int n = snprintf(pattern, sizeof(pattern), "\"%s\"", key);
	if (n <= 0 || n >= (int)sizeof(pattern)) {
		return NULL;
	}

	const char *p = strstr(json, pattern);
	if (!p)
		return NULL;

	// ':' 까지 이동
	p = strchr(p + n, ':');
	if (!p)
		return NULL;
	p++; // ':' 다음으로

	// 공백 스킵
	while (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n') {
		p++;
	}
	return p;
}

static bool json_get_bool(const char *json, const char *key, bool *out,
						  bool *present) {
	*present = false;
	const char *p = find_key(json, key);
	if (!p)
		return true; // key가 없는 건 에러 아님

	if (strncmp(p, "true", 4) == 0) {
		*out = true;
		*present = true;
		return true;
	} else if (strncmp(p, "false", 5) == 0) {
		*out = false;
		*present = true;
		return true;
	}
	ESP_LOGW(TAG, "Key '%s' found but not bool", key);
	return false;
}

static bool json_get_number(const char *json, const char *key, double *out,
							bool *present) {
	*present = false;
	const char *p = find_key(json, key);
	if (!p)
		return true; // 없음 = 에러 아님

	char *endptr = NULL;
	double v = strtod(p, &endptr);
	if (endptr == p) {
		ESP_LOGW(TAG, "Key '%s' found but not number", key);
		return false;
	}
	*out = v;
	*present = true;
	return true;
}

static bool json_get_string(const char *json, const char *key, char *out,
							size_t out_size, bool *present) {
	*present = false;
	if (out_size == 0) {
		return false;
	}

	const char *p = find_key(json, key);
	if (!p) {
		return true; // key 없음 = 에러 아님
	}

	if (*p != '"') {
		ESP_LOGW(TAG, "Key '%s' is not a JSON string", key);
		return false;
	}
	p++; // opening quote 넘어감

	size_t idx = 0;
	while (*p && *p != '"') {
		if (idx + 1 < out_size) {
			out[idx++] = *p;
		}
		p++;
	}
	out[idx] = '\0';

	if (*p != '"') {
		ESP_LOGW(TAG, "Key '%s' string not terminated", key);
		return false;
	}

	*present = true;
	return true;
}

static void desired_shadow_update_from_cmd(const hxb_cmd_desired_t *cmd) {
	// cmd_rev: 들어오면 그대로 덮어쓰고,
	// 안 들어왔는데 아직 초기화 안 됐으면 1 같은 디폴트 줄 수도 있음.
	if (cmd->has_cmd_rev) {
		s_desired_shadow.cmd_rev = (uint16_t)cmd->cmd_rev;
		s_desired_shadow.cmd_rev_initialized = true;
	} else if (!s_desired_shadow.cmd_rev_initialized) {
		s_desired_shadow.cmd_rev = 1;
		s_desired_shadow.cmd_rev_initialized = true;
	}

	// Digital
	if (cmd->has_PSU48_ENA) {
		s_desired_shadow.psu48_ena = cmd->PSU48_ENA;
	}
	if (cmd->has_ELECTROLYTE_CTRL_ENA) {
		s_desired_shadow.electrolyte_ctrl_ena = cmd->ELECTROLYTE_CTRL_ENA;
	}
	if (cmd->has_HIGHP_PRODUCTION_ENA) {
		s_desired_shadow.highp_production_ena = cmd->HIGHP_PRODUCTION_ENA;
	}

	// Analog
	if (cmd->has_PSU48_V_SET) {
		s_desired_shadow.psu48_v_set = cmd->PSU48_V_SET;
	}
	if (cmd->has_PSU48_I_SET) {
		s_desired_shadow.psu48_i_set = cmd->PSU48_I_SET;
	}
	if (cmd->has_ELECTROLYTE_T_SP_DEGC) {
		s_desired_shadow.electrolyte_t_sp_degC = cmd->ELECTROLYTE_T_SP_DEGC;
	}
	if (cmd->has_ELECTROLYTE_FLOW_SP_SLM) {
		s_desired_shadow.electrolyte_flow_sp_slm = cmd->ELECTROLYTE_FLOW_SP_SLM;
	}
}

/* Desired JSON 파서 */
static bool parse_desired(const char *json, int len, hxb_cmd_desired_t *out) {
	(void)len;
	memset(out, 0, sizeof(*out));

	// cmd_rev
	{
		double v;
		bool present;
		if (!json_get_number(json, "cmd_rev", &v, &present)) {
			return false;
		}
		if (present) {
			out->cmd_rev = (uint32_t)v;
			out->has_cmd_rev = true;
		}
	}

	// Digital
	json_get_bool(json, "PSU48_ENA", &out->PSU48_ENA, &out->has_PSU48_ENA);

	json_get_bool(json, "ELECTROLYTE_CTRL_ENA", &out->ELECTROLYTE_CTRL_ENA,
				  &out->has_ELECTROLYTE_CTRL_ENA);

	json_get_bool(json, "HIGHP_PRODUCTION_ENA", &out->HIGHP_PRODUCTION_ENA,
				  &out->has_HIGHP_PRODUCTION_ENA);

	// Analog
	{
		double v;
		bool present;
		if (!json_get_number(json, "PSU48_V_SET", &v, &present))
			return false;
		if (present) {
			out->PSU48_V_SET = (float)v;
			out->has_PSU48_V_SET = true;
		}
	}
	{
		double v;
		bool present;
		if (!json_get_number(json, "PSU48_I_SET", &v, &present))
			return false;
		if (present) {
			out->PSU48_I_SET = (float)v;
			out->has_PSU48_I_SET = true;
		}
	}
	{
		double v;
		bool present;
		if (!json_get_number(json, "ELECTROLYTE_T_SP_DEGC", &v, &present))
			return false;
		if (present) {
			out->ELECTROLYTE_T_SP_DEGC = (float)v;
			out->has_ELECTROLYTE_T_SP_DEGC = true;
		}
	}
	{
		double v;
		bool present;
		if (!json_get_number(json, "ELECTROLYTE_FLOW_SP_SLM", &v, &present))
			return false;
		if (present) {
			out->ELECTROLYTE_FLOW_SP_SLM = (float)v;
			out->has_ELECTROLYTE_FLOW_SP_SLM = true;
		}
	}

	return true;
}

/* Emergency JSON 파서 (cmd == "EMERGENCY_STOP" 만 체크) */
static bool parse_emergency(const char *json, int len,
							hxb_cmd_emergency_t *out) {
	(void)len;
	memset(out, 0, sizeof(*out));

	const char *p = find_key(json, "cmd");
	if (!p) {
		out->emergency = false;
		return true;
	}

	if (*p == '"') {
		p++;
		if (strncmp(p, "EMERGENCY_STOP", strlen("EMERGENCY_STOP")) == 0) {
			out->emergency = true;
		}
	} else {
		out->emergency = false;
	}
	return true;
}

/* Lease JSON 파서: {"ts":"2025-12-03T13:45:12.345Z", "seq_hb":123, ...} */
#include "driver/uart.h" // 혹시 아직 안 들어있으면 상단에 추가

// MQTT에서 받은 JSON 문자열(json, len)에서 "ts" 값을 찾아서
// ISO-8601 형태(YYYY-MM-DDTHH:MM:SS...)를 파싱해서 시각으로 뽑는다.
static bool parse_lease_ts(const char *json_raw, int len, int *year, int *month,
						   int *day, int *hour, int *minute, int *second) {
	// 안전하게 쓰기 위해 NULL-terminated 복사본 생성
	char *json = (char *)malloc(len + 1);
	if (!json) {
		return false;
	}
	memcpy(json, json_raw, len);
	json[len] = '\0';

	// "ts" 키워드 위치 찾기
	char *p = strstr(json, "\"ts\"");
	if (!p) {
		free(json);
		return false;
	}

	// ':' 까지 이동
	p = strchr(p, ':');
	if (!p) {
		free(json);
		return false;
	}

	// 첫 번째 따옴표(") 찾기
	p = strchr(p, '"');
	if (!p) {
		free(json);
		return false;
	}
	p++; // 따옴표 지나고 실제 문자열 시작

	// 여기부터 ISO-8601 예상: 2025-12-03T13:45:12.345Z ...
	int y, mo, d, h, mi, s;
	int n = sscanf(p, "%4d-%2d-%2dT%2d:%2d:%2d", &y, &mo, &d, &h, &mi, &s);

	if (n != 6) {
		free(json);
		return false;
	}

	// 간단 범위 체크 (필요하면 더 디테일하게)
	if (y < 2000 || y > 2099 || mo < 1 || mo > 12 || d < 1 || d > 31 || h < 0 ||
		h > 23 || mi < 0 || mi > 59 || s < 0 || s > 59) {
		free(json);
		return false;
	}

	*year = y;
	*month = mo;
	*day = d;
	*hour = h;
	*minute = mi;
	*second = s;

	free(json);
	return true;
}

static void uart_send_cmd_desired(const hxb_cmd_desired_t *cmd) {
	char line[128];

	uint16_t rev = s_desired_shadow.cmd_rev;
	int psu48 = s_desired_shadow.psu48_ena ? 1 : 0;
	int elec = s_desired_shadow.electrolyte_ctrl_ena ? 1 : 0;
	int highp = s_desired_shadow.highp_production_ena ? 1 : 0;

	float v_set = s_desired_shadow.psu48_v_set;
	float i_set = s_desired_shadow.psu48_i_set;
	float t_sp = s_desired_shadow.electrolyte_t_sp_degC;
	float flow_sp = s_desired_shadow.electrolyte_flow_sp_slm;

	int len =
		snprintf(line, sizeof(line), "CMD,%u,%d,%d,%d,%.3f,%.3f,%.3f,%.3f\r\n",
				 (unsigned)rev, psu48, elec, highp, (double)v_set,
				 (double)i_set, (double)t_sp, (double)flow_sp);

	if (len <= 0 || len >= (int)sizeof(line)) {
		ESP_LOGW(TAG,
				 "uart_send_cmd_desired_from_shadow: line truncated or error");
		return;
	}

	uart_write_bytes(STM32_UART_PORT, line, len);

	ESP_LOGI(TAG, "UART Desired (ASCII full) sent: %s", line);
}

/* Emergency → UART 패킹/송신
 *  [0] STX   = 0x02
 *  [1] TYPE  = 0x11  (Emergency)
 *  [2] LEN   = 1
 *  [3] EMG   = 0x01 (EMERGENCY_STOP)
 *  [4] CRC_L
 *  [5] CRC_H
 */
static void uart_send_cmd_emergency(const hxb_cmd_emergency_t *emg) {
	// 이번 설계에서는 EMERGENCY_STOP 요청일 때만 프레임을 보냄
	if (!emg->emergency) {
		// 필요하면 "EMG,0\r\n" 을 보내서 해제를 구현할 수도 있음
		return;
	}

	char line[16];
	int len = snprintf(line, sizeof(line), "EMG,1\r\n");
	if (len <= 0 || len >= (int)sizeof(line)) {
		ESP_LOGW(TAG, "uart_send_cmd_emergency: snprintf error");
		return;
	}

	uart_write_bytes(STM32_UART_PORT, line, len);

	ESP_LOGI(TAG, "UART Emergency (ASCII) sent: %s", line);
}
/* --------------------------------------------------------------------------
 *  MQTT 토픽 문자열 생성 헬퍼
 * -------------------------------------------------------------------------- */

static void make_topic(char *buf, size_t buf_len, const char *suffix) {
	// ex) suffix = "telemetry" → hxb2k/HXB2k-001/telemetry
	snprintf(buf, buf_len, "hxb2k/%s-%s/%s", HXB2_DEVICE_ID,
			 dev_uuid, suffix);
	ESP_LOGI(TAG, "%s\n", buf);
}

/* --------------------------------------------------------------------------
 *  Telemetry / Lease Task
 * -------------------------------------------------------------------------- */

// 1 Hz Telemetry: 더미 프레임 돌려가며 Publish
/*
static void telemetry_task(void *arg) {
	char topic[64];
	make_topic(topic, sizeof(topic), "telemetry");
	ESP_LOGI(TAG, "Telemetry task started, topic=%s", topic);

	int idx = 0;
	while (1) {
		if (s_mqtt == NULL) {
			vTaskDelay(pdMS_TO_TICKS(1000));
			continue;
		}

		const char *payload = hxb2_dummy_frames[idx];
		int len = (int)strlen(payload);

		int msg_id = esp_mqtt_client_publish(s_mqtt, topic, payload, len,
											 1, // QoS 1
											 0	// retain = 0
		);

		ESP_LOGI(TAG, "Telemetry[%d] published, msg_id=%d, len=%d", idx, msg_id,
				 len);

		// idx++;
		if (idx >= HXB2_DUMMY_FRAME_COUNT) {
			idx = 0; // 다시 처음부터 반복
		}

		vTaskDelay(1000 / portTICK_PERIOD_MS); // 1 Hz
	}
}*/

// Client lease heartbeat: 1 Hz
/*
static void lease_task(void *arg) {
	char topic[64];
	make_topic(topic, sizeof(topic), "client/lease");
	ESP_LOGI(TAG, "Lease task started, topic=%s", topic);

	char payload[128];

	while (1) {
		if (s_mqtt == NULL) {
			vTaskDelay(pdMS_TO_TICKS(1000));
			continue;
		}

		snprintf(payload, sizeof(payload),
				 "{\"schema\":\"hxb2k-icd:2.0\","
				 "\"device_id\":\"%s\","
				 "\"alive\":true}",
				 HXB2_DEVICE_ID);

		int len = (int)strlen(payload);

		int msg_id = esp_mqtt_client_publish(s_mqtt, topic, payload, len,
											 1, // QoS 1
											 0);

		ESP_LOGD(TAG, "Lease heartbeat published, msg_id=%d", msg_id);

		vTaskDelay(pdMS_TO_TICKS(1000)); // 1 Hz
	}
}*/

/* --------------------------------------------------------------------------
 *  Command 처리 (Desired / Emergency)
 * -------------------------------------------------------------------------- */

// Desired JSON 처리
static void handle_cmd_desired(const char *topic, const char *data, int len) {
	ESP_LOGI(TAG, "CMD_DESIRED: topic=%s, len=%d", topic, len);

	// MQTT payload는 NULL-terminated가 아니니까 복사해서 끝에 '\0' 추가
	char *json = (char *)malloc(len + 1);
	if (!json) {
		ESP_LOGE(TAG, "malloc failed in handle_cmd_desired");
		return;
	}
	memcpy(json, data, len);
	json[len] = '\0';

	ESP_LOGI(TAG, "CMD_DESIRED_STR: %s", json);

	hxb_cmd_desired_t cmd;
	if (!parse_desired(json, len, &cmd)) {
		ESP_LOGW(TAG, "parse_desired failed");
		free(json);
		return;
	}
	desired_shadow_update_from_cmd(&cmd);
	// UART 프레임으로 STM32에 전송
	uart_send_cmd_desired(&cmd);

	free(json);
}

// Emergency JSON 처리
static void handle_cmd_emergency(const char *topic, const char *data, int len) {
	ESP_LOGW(TAG, "CMD_EMERGENCY: topic=%s, len=%d", topic, len);

	char *json = (char *)malloc(len + 1);
	if (!json) {
		ESP_LOGE(TAG, "malloc failed in handle_cmd_emergency");
		return;
	}
	memcpy(json, data, len);
	json[len] = '\0';

	ESP_LOGW(TAG, "CMD_EMERGENCY_STR: %s", json);

	hxb_cmd_emergency_t emg;
	if (!parse_emergency(json, len, &emg)) {
		ESP_LOGW(TAG, "parse_emergency failed");
		free(json);
		return;
	}

	uart_send_cmd_emergency(&emg);

	free(json);
}
// Lease(JSON) 처리: 시간 동기 프레임 → STM32(UART1)로 ASCII 전송
static void handle_cmd_lease(const char *topic, const char *data, int len) {
	ESP_LOGI(TAG, "LEASE cmd: topic=%s, len=%d", topic, len);

	int year, month, day, hour, minute, second;

	if (!parse_lease_ts(data, len, &year, &month, &day, &hour, &minute,
						&second)) {
		ESP_LOGW(TAG, "Failed to parse 'ts' from lease JSON");
		return;
	}

	// ASCII 프로토콜: TIME,YYYY-MM-DD,HH:MM:SS\r\n
	char line[64];
	int n =
		snprintf(line, sizeof(line), "TIME,%04d-%02d-%02d,%02d:%02d:%02d\r\n",
				 year, month, day, hour, minute, second);
	if (n <= 0 || n >= (int)sizeof(line)) {
		ESP_LOGE(TAG, "TIME line snprintf error");
		return;
	}

	// UART1로 전송 (STM32랑 연결된 UART)
	// ESP-IDF 기준: UART_NUM_1 사용
	int written = uart_write_bytes(UART_NUM_1, line, n);
	if (written < 0) {
		ESP_LOGE(TAG, "UART write failed for TIME frame");
	} else {
		ESP_LOGI(TAG, "UART TIME frame sent: %s", line);
	}
}

/* --------------------------------------------------------------------------
 *  MQTT 이벤트 핸들러
 * -------------------------------------------------------------------------- */
void mqtt_device_topic_subscribe() {
	if (subscribed) {
		return;
	}
	char topic_desired[64];
	char topic_emerg[64];
	char topic_lease[64];

	make_topic(topic_desired, sizeof(topic_desired), "cmd/desired");
	make_topic(topic_emerg, sizeof(topic_emerg), "cmd/emergency");
	make_topic(topic_lease, sizeof(topic_lease), "client/lease");

	esp_mqtt_client_subscribe(s_mqtt, topic_desired, 1);
	esp_mqtt_client_subscribe(s_mqtt, topic_emerg, 1);
	esp_mqtt_client_subscribe(s_mqtt, topic_lease, 1);
	subscribed=true;
}
static void mqtt_event_handler(void *handler_args, esp_event_base_t base,
							   int32_t event_id, void *event_data) {
	esp_mqtt_event_handle_t event = (esp_mqtt_event_handle_t)event_data;

	switch ((esp_mqtt_event_id_t)event_id) {
	case MQTT_EVENT_CONNECTED: {
		ESP_LOGI(TAG, "MQTT connected");

		subscribed=false;
		// 필요한 토픽 Subscribe
		if (strcmp(dev_uuid,"ZZZZ")!=0) {
			mqtt_device_topic_subscribe();
		}
		g_mqtt_client = s_mqtt;

		// Telemetry / lease task는 한 번만 생성
		/*
		if (!s_telemetry_task_started) {
			xTaskCreate(telemetry_task, "telemetry_task",
						4096, NULL, 5, NULL);
			s_telemetry_task_started = true;
		}
		*/
		/*if (!s_lease_task_started) {
			xTaskCreate(lease_task, "lease_task",
						2048, NULL, 5, NULL);
			s_lease_task_started = true;
		}*/
		break;
	}

	case MQTT_EVENT_DISCONNECTED:
		ESP_LOGW(TAG, "MQTT disconnected");
		subscribed=false;
		g_mqtt_client=NULL;
		break;

	case MQTT_EVENT_DATA: {
		// 수신 데이터 처리
		char topic[128];
		int topic_len = event->topic_len < (int)sizeof(topic) - 1
							? event->topic_len
							: (int)sizeof(topic) - 1;
		memcpy(topic, event->topic, topic_len);
		topic[topic_len] = '\0';

		const char *data = event->data;
		int len = event->data_len;

		ESP_LOGI(TAG, "MQTT_EVENT_DATA: topic=%s, len=%d", topic, len);

		// 토픽 분기
		char topic_desired[64];
		char topic_emerg[64];
		char topic_lease[64];
		make_topic(topic_desired, sizeof(topic_desired), "cmd/desired");
		make_topic(topic_emerg, sizeof(topic_emerg), "cmd/emergency");
		make_topic(topic_lease, sizeof(topic_lease), "client/lease");
		if (strcmp(topic, topic_desired) == 0) {
			handle_cmd_desired(topic, data, len);
		} else if (strcmp(topic, topic_emerg) == 0) {
			handle_cmd_emergency(topic, data, len);
		} else if (strcmp(topic, topic_lease) == 0) {
			handle_cmd_lease(topic, data, len);
		}
		break;
	}

	default:
		break;
	}
}

/* --------------------------------------------------------------------------
 *  외부에서 호출하는 MQTT 시작 함수
 * -------------------------------------------------------------------------- */

// 이 함수를 "ETH IP 할당 완료" 시점에 한 번만 호출해주면 됨
void hxb2_mqtt_start(void) {
	if (s_mqtt != NULL) {
		ESP_LOGW(TAG, "MQTT already started");
		return;
	}

	esp_mqtt_client_config_t cfg = {
		.broker.address.uri = HXB2_MQTT_URI,
		// 필요하면 username/password, client_id 등 추가
	};

	s_mqtt = esp_mqtt_client_init(&cfg);
	if (s_mqtt == NULL) {
		ESP_LOGE(TAG, "esp_mqtt_client_init failed");
		return;
	}

	esp_mqtt_client_register_event(s_mqtt, ESP_EVENT_ANY_ID, mqtt_event_handler,
								   NULL);

	esp_err_t err = esp_mqtt_client_start(s_mqtt);
	if (err != ESP_OK) {
		ESP_LOGE(TAG, "esp_mqtt_client_start failed: %s", esp_err_to_name(err));
		s_mqtt = NULL;
		return;
	}

	ESP_LOGI(TAG, "MQTT start requested (uri=%s)", HXB2_MQTT_URI);
}
