/*
 * telemetry.c
 *
 *  Created on: Nov 28, 2025
 *      Author: c6h6
 */

#include "telemetry.h"
#include "Baremetal_Tasks.h"
#include <math.h>
#include "adc_mux_app.h"
#include "usart.h"
#include "main.h"
#include "rtc.h"
#include <stdio.h>

static uint32_t g_esp32_tlm_seq = 0;

/**
 * @brief ESP32로 ASCII 텔레메트리 한 줄 전송
 * 포맷:
 *  HX2K,T,seq,ms,
 *  T1x10,T2x10,T3x10,
 *  ADC1[0..7],ADC2[0..7],ADC3[0..7],
 *  FAN_RPM,PUMP_HZ,
 *  DI_TANK1..DI_TANK4,DI_CONTACT,DI_SWITCH,DI_DAC_RDY\r\n
 */
extern double fan_duty;
extern uint16_t pump_dac_volt;
extern uint16_t circ_dac_volt;
extern RTC_HandleTypeDef hrtc;   // 어딘가에 선언되어 있을 것

#ifndef STM32F4_UID_BASE
// STM32F4 (예: F401/F405/F407/F410/F411/F412/F413/F429 등) UID base
#define STM32F4_UID_BASE  (0x1FFF7A10UL)
#endif

static inline const uint32_t* stm32_uid_words(void)
{
	return (const uint32_t*) STM32F4_UID_BASE; // [0], [1], [2]
}

// 96-bit UID -> 32-bit mix
static uint32_t stm32f4_uid_mix32(void)
{
    const uint32_t *uid = stm32_uid_words();
    uint32_t w0 = uid[0];
    uint32_t w1 = uid[1];
    uint32_t w2 = uid[2];

    // 가벼운 mix (회전/xor/곱)
    uint32_t x = w0 ^ (w1 + 0x9E3779B9u) ^ (w2 + 0x7F4A7C15u);

    x ^= (w1 << 11) | (w1 >> (32 - 11));
    x ^= (w2 << 17) | (w2 >> (32 - 17));
    x ^= (w0 <<  3) | (w0 >> (32 -  3));

    x *= 0x45d9f3bu;
    x ^= x >> 16;
    x *= 0x45d9f3bu;
    x ^= x >> 16;

    return x;
}

// out은 최소 5바이트 필요 ("0A9Z" + '\0')
void stm32f4_uid_to_base36_4(char out[5])
{
    static const char alphabet[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    const uint32_t MOD = 36u * 36u * 36u * 36u; // 36^4 = 1679616

    uint32_t v = stm32f4_uid_mix32() % MOD;

    // Base36 4자리 (항상 4자 유지)
    out[4] = '\0';
    out[3] = alphabet[v % 36u]; v /= 36u;
    out[2] = alphabet[v % 36u]; v /= 36u;
    out[1] = alphabet[v % 36u]; v /= 36u;
    out[0] = alphabet[v % 36u];
}
static void ESP32_SendTelemetryASCII(void)
{
	char buf[512];
	char UUID[5] = {0,};
	int len = 0;

	uint32_t seq = g_esp32_tlm_seq++;
	uint32_t ms = HAL_GetTick();

	// ----- RTC에서 현재 시간 읽기 -----
	RTC_TimeTypeDef sTime =
	{ 0 };
	RTC_DateTypeDef sDate =
	{ 0 };

	// 반드시 Time 먼저, 그 다음 Date 호출 (HAL 규칙)
	HAL_RTC_GetTime(&hrtc, &sTime, RTC_FORMAT_BIN);
	HAL_RTC_GetDate(&hrtc, &sDate, RTC_FORMAT_BIN);

	uint16_t year = (uint16_t) (2000 + sDate.Year);  // 보통 2000 offset
	uint8_t month = sDate.Month;
	uint8_t day = sDate.Date;
	uint8_t hour = sTime.Hours;
	uint8_t min = sTime.Minutes;
	uint8_t sec = sTime.Seconds;

	// 온도 스케일링 (℃ x10 → 정수)
	int t1_i = (int) lroundf(g_t1 * 10.0f);
	int t2_i = (int) lroundf(g_t2 * 10.0f);
	int t3_i = (int) lroundf(g_t3 * 10.0f);

	// RPM / Hz → 음수 보호 + saturate
	uint32_t rpm = (PWM_FAN_RPM < 0.0f) ? 0U : (uint32_t) lroundf(PWM_FAN_RPM);
	uint32_t hz =
			(TANK_ADD_PUMP_FREQ < 0.0f) ?
					0U : (uint32_t) lroundf(TANK_ADD_PUMP_FREQ);

	// 헤더 + seq + ms + RTC 날짜/시간 + T1..T3
	stm32f4_uid_to_base36_4(UUID);
	len += snprintf(&buf[len], sizeof(buf) - len,
			"HX2K,T,%s,%lu,%lu,%04u-%02u-%02u,%02u:%02u:%02u,%d,%d,%d",
			UUID, (unsigned long) seq, (unsigned long) ms,
			(unsigned int) year, (unsigned int) month, (unsigned int) day,
			(unsigned int) hour, (unsigned int) min, (unsigned int) sec, t1_i,
			t2_i, t3_i);

	// ADC1 전체 채널
	for (int i = 0; i < ADC1_NUM_CH; i++)
	{
		if (len < (int) sizeof(buf))
			len += snprintf(&buf[len], sizeof(buf) - len, ",%u",
					(unsigned int) s_adc1_shadow[i]);
	}

	// ADC2 전체 채널
	for (int i = 0; i < ADC2_NUM_CH; i++)
	{
		if (len < (int) sizeof(buf))
			len += snprintf(&buf[len], sizeof(buf) - len, ",%u",
					(unsigned int) s_adc2_shadow[i]);
	}

	// ADC3 전체 채널
	for (int i = 0; i < ADC3_NUM_CH; i++)
	{
		if (len < (int) sizeof(buf))
			len += snprintf(&buf[len], sizeof(buf) - len, ",%u",
					(unsigned int) s_adc3_shadow[i]);
	}
	// SPI ADC 전체 채널
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len, ",%ld",
				shared_cache.volt_sen_raw_filt);
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len, ",%ld",
				shared_cache.curr_sen_raw_filt);
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len, ",%ld",
				shared_cache.h_gas_p_sen1_raw_filt);
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len, ",%ld",
				shared_cache.h_gas_p_sen2_raw_filt);

	// FAN_RPM, PUMP_HZ, fan duty(%), pump/circ DAC
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len, ",%lu,%lu,%lu,%u,%u",
				(unsigned long) rpm, (unsigned long) hz,
				(unsigned long) (fan_duty * 100.0), pump_dac_volt,
				circ_dac_volt);

	// DI 13개 (0/1)
	if (len < (int) sizeof(buf))
		len += snprintf(&buf[len], sizeof(buf) - len,
				",%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u\r\n",
				(unsigned int) !DIN_TANK1(), (unsigned int) !DIN_TANK2(),
				(unsigned int) !DIN_TANK3(), (unsigned int) !DIN_TANK4(),
				(unsigned int) !DIN_CONTACT(),
				(unsigned int) shared_cache.is_emergency,
				(unsigned int) DIN_DAC_RDY(), (unsigned int) DIN_CONTACTCTL(),
				(unsigned int) DIN_H_GAS_SOL1_OUT(),
				(unsigned int) DIN_H_GAS_SOL2_OUT(),
				(unsigned int) DIN_H_GAS_SOL3_OUT(),
				(unsigned int) DIN_PSUENCTL(),
				(unsigned int) DIN_CARTRIDGE_HEATER());

	// TX (블로킹, 5 s 타임아웃)
	if (len > 0 && len < (int) sizeof(buf))
	{
		HAL_UART_Transmit(&huart1, (uint8_t*) buf, (uint16_t) len, 5000);
	}
}

/**
 * @brief 10Hz 주기로 위 텔레메트리 전송
 * main while(1) 루프에서 매 번 호출
 */
void ESP32_Telemetry_Task_1Hz(void)
{
	ESP32_SendTelemetryASCII();

}
