#include "out_ctl.h"

// === ioc 매핑: 각 라벨 → 실제 포트/핀 ===
// (STM32CubeMX 라벨/핀: ioc 기준)  :contentReference[oaicite:1]{index=1}
typedef struct {
  GPIO_TypeDef* port;
  uint16_t      pin;
  uint8_t       active_low; // 회로가 "Low=ON" 이면 1, "High=ON" 이면 0
} OutPin;

static const OutPin s_map[OUT_COUNT] = {
  [OUT_CARTRIDGE_HEATER] = { GPIOH, GPIO_PIN_2,  0 }, // PH2
  [OUT_H_GAS_SOL3]       = { GPIOH, GPIO_PIN_3,  0 }, // PH3
  [OUT_H_GAS_SOL2]       = { GPIOH, GPIO_PIN_4,  0 }, // PH4
  [OUT_H_GAS_SOL1]       = { GPIOH, GPIO_PIN_5,  0 }, // PH5
  [OUT_DC_FAN]           = { GPIOE, GPIO_PIN_7,  0 }, // PE7
  [OUT_EXT_FAN1]         = { GPIOA, GPIO_PIN_8,  0 }, // PA8
  [OUT_EXT_FAN2]         = { GPIOC, GPIO_PIN_9,  0 }, // PC9
  [OUT_PSU_ON]           = { GPIOF, GPIO_PIN_0,  0 }, // PF0
  [OUT_ELEC_CONTACT]     = { GPIOF, GPIO_PIN_2,  0 }, // PF2
  [OUT_SPARE_DO1]        = { GPIOG, GPIO_PIN_2,  0 }, // PG2
  [OUT_SPARE_DO2]        = { GPIOD, GPIO_PIN_15, 0 }, // PD15
  [OUT_SPARE_DO3]        = { GPIOD, GPIO_PIN_14, 0 }, // PD14
  [OUT_LED1]             = { GPIOG, GPIO_PIN_6,  0 }, // PG6
  [OUT_LED2]             = { GPIOG, GPIO_PIN_7,  0 }, // PG7
  [OUT_LED3]             = { GPIOG, GPIO_PIN_8,  0 } // PG8
};

static inline void write_active(OutChannel ch, GPIO_PinState on)
{
  if (ch >= OUT_COUNT) return;
  const OutPin* p = &s_map[ch];
  GPIO_PinState phys =
      (p->active_low ? (on ? GPIO_PIN_RESET : GPIO_PIN_SET)
                     : (on ? GPIO_PIN_SET   : GPIO_PIN_RESET));
  HAL_GPIO_WritePin(p->port, p->pin, phys);
}

void OUTCTL_Init(void)
{
  // 안전 상태로 초기화(필요 시 채널별로 ON/OFF 변경)
  for (int i = 0; i < OUT_COUNT; ++i) {
    write_active((OutChannel)i, 0); // 기본 OFF
  }
}

void OUTCTL_On(OutChannel ch)      { write_active(ch, 1); }
void OUTCTL_Off(OutChannel ch)     { write_active(ch, 0); }

void OUTCTL_Toggle(OutChannel ch)
{
  if (ch >= OUT_COUNT) return;
  GPIO_PinState cur = HAL_GPIO_ReadPin(s_map[ch].port, s_map[ch].pin);
  // 물리 상태 반전 → 논리 상태 토글
  HAL_GPIO_WritePin(s_map[ch].port, s_map[ch].pin,
                    (cur == GPIO_PIN_SET) ? GPIO_PIN_RESET : GPIO_PIN_SET);
}

void OUTCTL_Pulse(OutChannel ch, uint32_t ms)
{
  OUTCTL_On(ch);
  HAL_Delay(ms);
  OUTCTL_Off(ch);
}

void OUTCTL_BlinkAll(uint32_t period_ms)
{
  static uint32_t last = 0;
  uint32_t now = HAL_GetTick();
  if (now - last >= period_ms) {
    last = now;
    for (int i = 0; i < OUT_COUNT; ++i) OUTCTL_Toggle((OutChannel)i);
  }
}
