/*
 * hxb_cmd_stm32.c
 *
 *  Created on: Dec 3, 2025
 *      Author: c6h6
 */

/*
 * hxb_cmd_stm32.c
 *
 *  STM32-side parser for ASCII command protocol from ESP32.
 *
 *  Protocol (per line, terminated by "\r\n"):
 *
 *    TIME,YYYY-MM-DD,HH:MM:SS
 *    CMD,<rev>,<psu48>,<elec_ena>,<highp_ena>,<psu48_v>,<psu48_i>,<t_sp>,<flow_sp>
 *    EMG,<value>
 *
 *  Created on: Dec 3, 2025
 *      Author: c6h6
 */

#include "hxb2_cmd_stm32.h"

#include <string.h>
#include <stdio.h>

#include "rtc.h"
uint8_t g_rx_byte;

/* 내부 상태 포인터 */
hxb2_cmd_state_t g_hxb2_cmd_state;

/* 간단한 ASCII 라인 버퍼 */
#define HXB2_CMD_LINE_BUF_SIZE   128

static char s_line_buf[HXB2_CMD_LINE_BUF_SIZE];
static uint16_t s_line_len = 0;

/* 내부 함수 선언 */
static void HXB2_CMD_HandleLine(const char *line);


extern RTC_HandleTypeDef hrtc;
extern hxb2_cmd_state_t g_hxb2_cmd_state;

void HXB2_ApplyTimeToRTC(void)
{
    RTC_TimeTypeDef sTime = {0};
    RTC_DateTypeDef sDate = {0};

    uint16_t year  = g_hxb2_cmd_state.time_year;
    uint8_t  month = g_hxb2_cmd_state.time_month;
    uint8_t  day   = g_hxb2_cmd_state.time_day;
    uint8_t  hour  = g_hxb2_cmd_state.time_hour;
    uint8_t  min   = g_hxb2_cmd_state.time_minute;
    uint8_t  sec   = g_hxb2_cmd_state.time_second;

    /* HAL RTC는 연도를 0~99로 넣어야 함 (보통 2000년 기준) */
    uint8_t rtc_year = 0;
    if (year >= 2000) {
        rtc_year = (uint8_t)(year - 2000);
    } else {
        rtc_year = (uint8_t)year;  // 혹시 20xx 아닌 값이 올 경우 그냥 잘라서
    }

    /* 날짜 설정 */
    sDate.Year    = rtc_year;  // 0~99
    sDate.Month   = month;     // 1~12
    sDate.Date    = day;       // 1~31
    sDate.WeekDay = 0;         // 요일은 계산 안 하고 무시 (필요하면 따로 계산해서 넣어도 됨)

    /* 시간 설정 (24시간제) */
    sTime.Hours   = hour;      // 0~23
    sTime.Minutes = min;       // 0~59
    sTime.Seconds = sec;       // 0~59
    sTime.SubSeconds     = 0;
    sTime.DayLightSaving = RTC_DAYLIGHTSAVING_NONE;
    sTime.StoreOperation = RTC_STOREOPERATION_RESET;

    /* BIN 포맷으로 넣으면 BCD 변환 신경 안 써도 됨 */
    if (HAL_RTC_SetTime(&hrtc, &sTime, RTC_FORMAT_BIN) != HAL_OK)
    {
        // 에러 처리 (필요시 breakpoint, 로그 등)
        return;
    }

    if (HAL_RTC_SetDate(&hrtc, &sDate, RTC_FORMAT_BIN) != HAL_OK)
    {
        // 에러 처리
        return;
    }
}


/* 초기화 */
void HXB2_CMD_Init()
{
	memset(&g_hxb2_cmd_state, 0, sizeof(g_hxb2_cmd_state));
	s_line_len = 0;
}

/* 어플리케이션에서 상태를 보고 싶을 때 */
hxb2_cmd_state_t* HXB2_CMD_GetState(void)
{
	return &g_hxb2_cmd_state;
}

/* UART 바이트 단위 수신 핸들러
 *
 * ESP32 → STM32로 '\r\n' 끝나는 ASCII 문자열이 들어온다고 가정.
 */
void HXB2_CMD_OnRxByte(uint8_t byte)
{
	g_hxb2_cmd_state.tick_lastbytereceived=HAL_GetTick();
	if (byte == '\n')
	{
		/* 라인 종료: NULL-terminate */
		if (s_line_len < HXB2_CMD_LINE_BUF_SIZE)
		{
			s_line_buf[s_line_len] = '\0';
		}
		else
		{
			s_line_buf[HXB2_CMD_LINE_BUF_SIZE - 1] = '\0';
		}

		/* CR 제거 (있다면) */
		if (s_line_len > 0 && s_line_buf[s_line_len - 1] == '\r')
		{
			s_line_buf[s_line_len - 1] = '\0';
		}

		/* 한 줄 처리 */
		HXB2_CMD_HandleLine(s_line_buf);

		/* 다음 라인을 위해 버퍼 리셋 */
		s_line_len = 0;
		return;
	}

	/* '\r'는 무시 (라인 종료는 '\n' 기준) */
	if (byte == '\r')
	{
		return;
	}

	/* 버퍼에 축적 */
	if (s_line_len < (HXB2_CMD_LINE_BUF_SIZE - 1))
	{
		s_line_buf[s_line_len++] = (char) byte;
	}
	else
	{
		/* overflow 나면 이 라인은 버리고 다음 '\n'까지 버린다 */
		s_line_len = 0;
	}
}

/* 한 줄 단위 ASCII 명령 처리 */
static void HXB2_CMD_HandleLine(const char *line)
{
	if (!line)
		return;

	/* 앞쪽 공백 제거 */
	while (*line == ' ' || *line == '\t')
	{
		++line;
	}

	if (*line == '\0')
		return;

	/* TIME,YYYY-MM-DD,HH:MM:SS */
	if (strncmp(line, "TIME,", 5) == 0)
	{
		int year, month, day, hour, minute, second;
		int n = sscanf(line + 5, "%d-%d-%d,%d:%d:%d", &year, &month, &day,
				&hour, &minute, &second);
		if (n == 6)
		{
			/* 간단한 범위 체크 */
			if (year >= 2000 && year <= 2099 && month >= 1 && month <= 12
					&& day >= 1 && day <= 31 && hour >= 0 && hour <= 23
					&& minute >= 0 && minute <= 59 && second >= 0
					&& second <= 59)
			{
				g_hxb2_cmd_state.time_year = (uint16_t) year;
				g_hxb2_cmd_state.time_month = (uint8_t) month;
				g_hxb2_cmd_state.time_day = (uint8_t) day;
				g_hxb2_cmd_state.time_hour = (uint8_t) hour;
				g_hxb2_cmd_state.time_minute = (uint8_t) minute;
				g_hxb2_cmd_state.time_second = (uint8_t) second;
				g_hxb2_cmd_state.time_valid = true;
				HXB2_ApplyTimeToRTC();
			}
		}
		return;
	}

	/* CMD,<rev>,<psu48>,<elec_ena>,<highp_ena>,<psu48_v>,<psu48_i>,<t_sp>,<flow_sp> */
	if (strncmp(line, "CMD,", 4) == 0)
	{
		uint16_t cmd_rev;
		int d_psu = 0;
		int d_elec = 0;
		int d_highp = 0;
		float v_set = 0.0f;
		float i_set = 0.0f;
		float t_sp = 0.0f;
		float flow_sp = 0.0f;

		/* 모든 인자가 다 오지 않아도 되도록, sscanf 결과 개수에 따라 처리 */
		int n = sscanf(line + 4, "%hu,%d,%d,%d,%f,%f,%f,%f", &cmd_rev, &d_psu,
				&d_elec, &d_highp, &v_set, &i_set, &t_sp, &flow_sp);

		if (n >= 1)
		{
			g_hxb2_cmd_state.cmd_rev = cmd_rev;
			g_hxb2_cmd_state.cmd_rev_valid = true;
		}
		if (n >= 2)
		{
			g_hxb2_cmd_state.psu48_ena = (d_psu != 0);
			g_hxb2_cmd_state.psu48_ena_valid = true;
		}
		if (n >= 3)
		{
			g_hxb2_cmd_state.electrolyte_ctrl_ena = (d_elec != 0);
			g_hxb2_cmd_state.electrolyte_ctrl_ena_valid = true;
		}
		if (n >= 4)
		{
			g_hxb2_cmd_state.highp_production_ena = (d_highp != 0);
			g_hxb2_cmd_state.highp_production_ena_valid = true;
		}
		if (n >= 5)
		{
			g_hxb2_cmd_state.psu48_v_set = v_set;
			g_hxb2_cmd_state.psu48_v_set_valid = true;
		}
		if (n >= 6)
		{
			g_hxb2_cmd_state.psu48_i_set = i_set;
			g_hxb2_cmd_state.psu48_i_set_valid = true;
		}
		if (n >= 7)
		{
			g_hxb2_cmd_state.electrolyte_t_sp_degC = t_sp;
			g_hxb2_cmd_state.electrolyte_t_sp_valid = true;
		}
		if (n >= 8)
		{
			g_hxb2_cmd_state.electrolyte_flow_sp_slm = flow_sp;
			g_hxb2_cmd_state.electrolyte_flow_sp_valid = true;
		}

		return;
	}

	/* EMG,<value>  (value != 0 이면 Emergency latch) */
	if (strncmp(line, "EMG,", 4) == 0)
	{
		int val = 0;
		if (sscanf(line + 4, "%d", &val) == 1)
		{
			if (val != 0)
			{
				g_hxb2_cmd_state.emergency_stop_latched = true;
			}
			else
			{
				/* 필요하다면 0일 때 해제 로직도 추가할 수 있음 */
				// g_hxb2_cmd_state.emergency_stop_latched = false;
			}
		}
		return;
	}

	/* 그 외의 라인은 무시 */
}
