/*
 * Baremetal_Tasks.c
 *
 * Updated on: 2025. 12. 01.
 * Author: Firmware Dev
 * Description: Control Logic implementation matching user-defined SharedCache_t
 */

#include "Baremetal_Tasks.h"
#include <math.h>
#include <string.h>
#include <stdio.h>

// --- Hardware Drivers ---
#include "telemetry.h"
#include "main.h"
#include "adc_mux_app.h"
#include "max31865.h"
#include "out_ctl.h"
#include "ws2812.h"
#include "i2c.h"
#include "tim.h"
#include "ws2812.h"

#include "hxb2_cmd_stm32.h"

// =============================================================
// 0. 전역 변수 및 상수 설정
// =============================================================

volatile SharedCache_t shared_cache;
extern Max31865_t rtd1, rtd2, rtd3;

// Task 주기 (ms)
#define PERIOD_TASK_A_SENSORS   5   // 10Hz
#define PERIOD_TASK_B_REFILL    2   // 2Hz
#define PERIOD_TASK_C_CIRC      100   // 10Hz (Flow Control)
#define PERIOD_TASK_D_TEMP      500   // 2Hz
#define PERIOD_TASK_E_PSU       500    // 2Hz
#define PERIOD_TASK_F       500   // 100Hz
#define PERIOD_SUPERVISOR       1000  // 1Hz

// 타이머 변수
#ifndef TASKS_AS_FREERTOS
static uint32_t tick_A = 0, tick_B = 0, tick_C = 0, tick_D = 0, tick_E = 0,
		tick_F = 0, tick_Sup = 0;
#endif
// DAC 채널 매핑 (회로도 기준)
#define DAC_CH_PSU_CURRENT  0  // DAC1 (A) -> Task E
#define DAC_CH_FAN_SPEED    1  // DAC2 (B) -> Task D
#define DAC_CH_REFILL_PUMP  2  // DAC3 (C) -> Task B

// 제어 상수
#define REFILL_PUMP_DAC_VAL 39321  // 약 3V 출력 (보충 펌프 구동)
#define PI_KP_FLOW          0.15f  // 순환 펌프 P게인
#define PI_KI_FLOW          0.1f  // 순환 펌프 I게인

double fan_duty = 0;
uint16_t pump_dac_volt, circ_dac_volt;

float adc_to_mA(uint16_t adc_raw)
{
	const float VREF = 3.3f;
	const float R_SHUNT = 150.0f; // R159

	float v_adc = (adc_raw * VREF) / 4095.0f;
	float i_mA = (v_adc / R_SHUNT) * 1000.0f;

	return i_mA;
}

void FAN_SetPWM(double duty_ratio, double freq_hz)
{
	//HAL_TIM_PWM_Stop(&htim1, TIM_CHANNEL_2);
	//htim1.Instance->CNT=0;
	if (duty_ratio < 0.01f)
		duty_ratio = 0.01f;
	if (duty_ratio >= 0.99f)
		duty_ratio = 0.99f;
	if (freq_hz < 10.0f)
		freq_hz = 10.0f;   // 理쒖냼 10Hz �젣�븳

	const float tim_clk = 180000000.0f; // TIM1 clock
	uint32_t real_psc = 179; // 1MHz로 다운 (180MHz / 180)
	float tick_hz = tim_clk / (real_psc + 1); // 1,000,000 Hz

	// ARR = 1,000,000 / 2000 - 1 = 499 (해상도 500단계 - 충분함)
	uint32_t arr = (uint32_t) ((tick_hz / freq_hz) - 1.0f);

	uint32_t ccr = (uint32_t) ((arr + 1) * duty_ratio);

	__HAL_TIM_SET_PRESCALER(&htim1, real_psc);
	__HAL_TIM_SET_AUTORELOAD(&htim1, arr);
	__HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_2, ccr);
	/*if (duty_ratio == 0.0f)
	 {
	 //HAL_TIM_PWM_Stop(&htim1, TIM_CHANNEL_2);
	 //__HAL_TIM_SET_COMPARE(&htim1, TIM_CHANNEL_2, 2);
	 //htim1.Instance->CNT=0;
	 //OUTCTL_Off(OUT_DC_FAN);
	 }
	 else
	 {
	 HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_2);
	 //OUTCTL_On(OUT_DC_FAN);
	 }*/
}

// =============================================================
// 1. 초기화 (System_Logic_Init)
// =============================================================

static void Apply_Cmd_Once_From_HMI(void)
{

	if (g_hxb2_cmd_state.time_valid == true)
	{
		// (옵션) TIME 명령으로 RTC를 맞춘 뒤, SharedCache의 time_sync_ok / last_sync_time을
		//        여기서 처리하고 싶으면, TIME 처리 코드에서 아래처럼 세트해도 됨:
		//
		shared_cache.time_sync_ok = true;
		g_hxb2_cmd_state.time_valid = false;
		shared_cache.last_sync_time = HAL_GetTick();
	}
	// 1) PSU Enable (48V 전원 ON/OFF)
	if (g_hxb2_cmd_state.psu48_ena_valid)
	{
		shared_cache.psu_enable = g_hxb2_cmd_state.psu48_ena;
		g_hxb2_cmd_state.psu48_ena_valid = false;
	}

	// 2) Electrolyte Control Enable (전해질 제어 전체 ON/OFF)
	if (g_hxb2_cmd_state.electrolyte_ctrl_ena_valid)
	{
		shared_cache.electrolyte_enable = g_hxb2_cmd_state.electrolyte_ctrl_ena;
		g_hxb2_cmd_state.electrolyte_ctrl_ena_valid = false;
	}

	// 3) High Pressure Mode (고압 모드 플래그)
	if (g_hxb2_cmd_state.highp_production_ena_valid)
	{
		shared_cache.high_pressure_mode = g_hxb2_cmd_state.highp_production_ena;
		if (shared_cache.high_pressure_mode == true)
		{
			shared_cache.valve_mode = VALVE_MODE_HIGHP;
		}
		else
		{
			shared_cache.valve_mode = VALVE_MODE_ATMOS;
		}
		g_hxb2_cmd_state.highp_production_ena_valid = false;
	}

	// 4) Set Values (SV) ─ PSU 전압/전류 세트포인트
	if (g_hxb2_cmd_state.psu48_v_set_valid)
	{
		shared_cache.voltage_sv = g_hxb2_cmd_state.psu48_v_set;   // [V]
		g_hxb2_cmd_state.psu48_v_set_valid = false;
	}

	if (g_hxb2_cmd_state.psu48_i_set_valid)
	{
		shared_cache.current_sv = g_hxb2_cmd_state.psu48_i_set;   // [A]
		g_hxb2_cmd_state.psu48_i_set_valid = false;
	}

	// 5) Set Values (SV) ─ 전해질 온도 / 유량 세트포인트
	if (g_hxb2_cmd_state.electrolyte_t_sp_valid)
	{
		shared_cache.temp_sv = g_hxb2_cmd_state.electrolyte_t_sp_degC;  // [°C]
		g_hxb2_cmd_state.electrolyte_t_sp_valid = false;
	}

	if (g_hxb2_cmd_state.electrolyte_flow_sp_valid)
	{
		shared_cache.flow_sv = g_hxb2_cmd_state.electrolyte_flow_sp_slm; // [SLM]
		g_hxb2_cmd_state.electrolyte_flow_sp_valid = false;
	}

	// 6) Emergency (CMD/EMG에서 들어온 비상 정지 래치 → shared_cache에 반영)
	if (g_hxb2_cmd_state.emergency_stop_latched && !shared_cache.is_emergency)
	{
		shared_cache.is_emergency = true;
		// g_hxb2_cmd_state.emergency_stop_latched 는 그대로 두고,
		// shared_cache 쪽만 래치해둔다 (한 번 STOP 들어오면 계속 유지).
	}
}

void System_Logic_Init(void)
{
	// 1. 구조체 초기화
	memset((void*) &shared_cache, 0, sizeof(SharedCache_t));

	// 2. 기본값 설정 (HMI 연결 전 기본 동작 값)
	shared_cache.is_emergency = false;
	shared_cache.temp_sv = 20.0f;     // 목표 온도 25도
	shared_cache.current_sv = 0.0f;   // 초기 전류 0A
	shared_cache.voltage_sv = 38.0f;  // 초기 전압 48V
	shared_cache.flow_sv = 1.5f;      // 초기 유량 1.0 LPM (가정)

	shared_cache.electrolyte_enable = false;
	shared_cache.psu_enable = false;
	shared_cache.valve_mode = VALVE_MODE_ATMOS;
	//shared_cache.pump_refill_on = true;

	// 3. 하드웨어 초기화
	//MCP4728_Write_AllChannels_Same(&hi2c3, 0);

	OUTCTL_On(OUT_DC_FAN);
	osDelay(1000);
	HAL_TIM_PWM_Start(&htim1, TIM_CHANNEL_2);
	ws2812Init();
	//OUTCTL_On(OUT_PSU_ON);
	//OUTCTL_On(OUT_H_GAS_SOL1);
	//OUTCTL_On(OUT_H_GAS_SOL2);
	//OUTCTL_On(OUT_H_GAS_SOL3);
	//All_Valves_Off(); // 밸브 잠금

#ifdef TASKS_AS_FREERTOS

#endif

}

// =============================================================
// 2. 개별 Task 구현
// =============================================================

// [Task A] 센서 데이터 수집 -> PV 업데이트
void Task_A_Sensors(void)
{
	// 1. ADC Mux 스캔 및 물리량 변환
	ADC_APP_Poll();

	// 2. RTD 온도 읽기 -> elect rolyte_temp 업데이트
	RTD_Update_All();
	Max31865_readTempC(&rtd2, &shared_cache.electrolyte_temp);

	// 3. 수위 센서 (GPIO) -> Digital Inputs 업데이트
	// Active Low (GND 연결 시 감지) 가정
	shared_cache.level_sensor_high = !DIN_TANK2();
	shared_cache.level_sensor_low = !DIN_TANK1();

	// 4. Emergency 스위치 입력: true로 "변할 때" 단 한 번만 래치
	SENSOR_FreqTask();
	if (DIN_SWITCH() == false)
	{
		HAL_NVIC_SystemReset();
	}
	/*{
	 static bool emergency_latched_from_switch = false;
	 bool emergency_now = !DIN_SWITCH();   // Active Low

	 if (emergency_now && !emergency_latched_from_switch)
	 {
	 shared_cache.is_emergency = true;
	 emergency_latched_from_switch = true;
	 }
	 // false로 돌아갈 때 해제하고 싶으면 else-if 추가
	 }*/

	// 5. CMD로 들어온 파라미터들을 "한 번만" shared_cache에 적용
	Apply_Cmd_Once_From_HMI();
	Task_Supervisor();
}

// [Task B] 전해질 보충 (DAC3)
void Task_B_Refill(void)
{
	static uint8_t state = 0;
	uint16_t dac_val = 0;

	if (!shared_cache.electrolyte_enable || shared_cache.is_emergency)
	{
		shared_cache.pump_refill_on = false;

		// DAC8554 D채널(=3) 0 출력
		(void) DAC_SetCode(3, 0x0000, 5 /*ms*/);

		state = 0;
		pump_dac_volt = 0;
		return;
	}

	switch (state)
	{
	case 0:
		if (!DIN_TANK1() == true)
		{
			shared_cache.pump_refill_on = true;
			state = 1;
		}
		break;

	case 1:
		if (!DIN_TANK2() == false)
		{
			shared_cache.pump_refill_on = false;
			state = 0;
		}
		break;

	default:
		break;
	}

	if (shared_cache.pump_refill_on)
		dac_val = (uint16_t) REFILL_PUMP_DAC_VAL; // 16-bit 값이라고 가정
	else
		dac_val = 0;

	pump_dac_volt = dac_val;

	// DAC8554 D채널(=3)로 즉시 업데이트
	(void) DAC_SetCode(3, pump_dac_volt, 5 /*ms*/);
}

// [Task C] 전해질 순환 (PI 제어) -> pump_circulation_duty 계산 / DAC_B
#include <math.h>   // fabsf

#define FLOW_MAX_MEAS        5.7f   // 실측된 최대 유량
#define FLOW_DEADBAND        0.005f  // ±0.05 LPM 이내 무시
#define FLOW_I_MAX           10.0f
#define FLOW_I_MIN          -10.0f
#define FLOW_MIN_CMD         0.08f  // 펌프 최소 duty (필요시 조정)
#define FLOW_CTRL_DT         0.001f   // Task_C 주기 (100ms 가정)

void Task_C_Circulation(void)
{
	// 유량 로우패스 필터 상태
	static float flow_pv_filt = 0.0f;
	static uint8_t filt_initialized = 0;

	if (!shared_cache.electrolyte_enable || shared_cache.is_emergency)
	{
		shared_cache.pump_circulation_duty = 0.0f;
		shared_cache.flow_pid_integral = 0.0f;

		// DAC 출력 0으로 (채널 C=2 가정)
		(void) DAC_SetCode(2, 0x0000, 5 /*ms*/);

		circ_dac_volt = 0;
		filt_initialized = 0;
		flow_pv_filt = 0.0f;
		return;
	}

	// 1) 현재 유량 RAW 읽기 (4~20 mA -> LPM)
	float raw_flow = ((adc_to_mA(s_adc1_shadow[ADC1_TANK_FLOW_ADC]) - 4.0f)
			* 9.5f / 16.0f);

	// 2) 1차 로우패스 필터
	if (!filt_initialized)
	{
		flow_pv_filt = raw_flow;
		filt_initialized = 1;
	}
	else
	{
		const float alpha = 0.2f; // 0.1~0.3에서 튜닝 가능
		flow_pv_filt += alpha * (raw_flow - flow_pv_filt);
	}

	float current_flow = flow_pv_filt;

	// 3) SV 클램프 (물리적으로 가능한 범위로)
	float sp_cmd = shared_cache.flow_sv;   // HMI에서 들어온 값
	if (sp_cmd < 0.0f)
		sp_cmd = 0.0f;
	if (sp_cmd > FLOW_MAX_MEAS)
		sp_cmd = FLOW_MAX_MEAS;

	float sp = sp_cmd;

	// 4) 피드포워드 duty (대략 linear 가정: 0~5.7 LPM -> 0~1.0 duty)
	float duty_ff = 0.0f;
	if (FLOW_MAX_MEAS > 0.0f)
		duty_ff = sp / FLOW_MAX_MEAS;

	if (duty_ff > 1.0f)
		duty_ff = 1.0f;
	if (duty_ff < 0.0f)
		duty_ff = 0.0f;

	// 5) SP 기준 게인 스케줄링
	float gain_scale;
	if (sp < 1.0f)
		gain_scale = 0.3f;
	else if (sp < 2.0f)
		gain_scale = 0.5f;
	else if (sp < 4.0f)
		gain_scale = 0.8f;
	else
		gain_scale = 1.0f;

	float kp = PI_KP_FLOW * gain_scale;
	float ki = PI_KI_FLOW * gain_scale;

	// 6) PI 제어 (피드포워드 + PI)
	float error = sp - current_flow;

	if (fabsf(error) < FLOW_DEADBAND)
		error = 0.0f;

	float dt = FLOW_CTRL_DT;

	float integrator = shared_cache.flow_pid_integral;

	float u_pi_unsat = kp * error + ki * integrator;
	float output_unsat = duty_ff + u_pi_unsat;

	bool can_integrate = true;
	if (output_unsat >= 1.0f && error > 0.0f)
		can_integrate = false;
	else if (output_unsat <= 0.0f && error < 0.0f)
		can_integrate = false;

	if (can_integrate && error != 0.0f)
	{
		integrator += (error * dt);

		if (integrator > FLOW_I_MAX)
			integrator = FLOW_I_MAX;
		if (integrator < FLOW_I_MIN)
			integrator = FLOW_I_MIN;
	}

	shared_cache.flow_pid_integral = integrator;

	float u_pi = kp * error + ki * integrator;
	float output = duty_ff + u_pi;

	if (output > 1.0f)
		output = 1.0f;
	if (output < 0.0f)
		output = 0.0f;

	if (output > 0.0f && output < FLOW_MIN_CMD)
		output = FLOW_MIN_CMD;

	// 7) Duty -> DAC (16-bit)
	shared_cache.pump_circulation_duty = output;

	// 0.0~1.0 -> 0~65535
	uint16_t dac_out = (uint16_t) (output * 65535.0f);
	circ_dac_volt = dac_out;

	// DAC8554 채널 C(=2)로 즉시 업데이트 (큐 통해 직렬화)
	(void) DAC_SetCode(2, circ_dac_volt, 5 /*ms*/);
}

// [Task D] 온도 제어 (DAC2 - Fan)
#define TEMPCTRL_FAN_DUTY_LOW   0.2f
#define TEMPCTRL_FAN_DUTY_MID   0.4f

void Task_D_TempControl(void)
{
	if (!shared_cache.electrolyte_enable || shared_cache.is_emergency)
	{
		shared_cache.fan_on = false;
		shared_cache.heater_on = false;
		fan_duty = 0.0f;
		FAN_SetPWM(0.0f, 2500);
		OUTCTL_Off(OUT_CARTRIDGE_HEATER);
		return;
	}

	float pv = shared_cache.electrolyte_temp;
	float sv = shared_cache.temp_sv;

	// 1) 히터 제어 (기존 유지)
	if (pv < sv - 2.0f)
	{
		shared_cache.heater_on = true;
		OUTCTL_On(OUT_CARTRIDGE_HEATER);
	}
	else
	{
		shared_cache.heater_on = false;
		OUTCTL_Off(OUT_CARTRIDGE_HEATER);
	}

	float diff = pv - (sv - 0.5f);   // == pv - sv + 1.0f
	double duty = 0.0f;

	if (diff <= TEMPCTRL_FAN_DUTY_LOW)
	{
		duty = 0.0f;
		shared_cache.fan_on = false;
	}
	else if (diff < 0.5f)
	{
		// 0.2..0.5 => 0..0.4
		duty = ((diff - TEMPCTRL_FAN_DUTY_LOW) / (0.5f - TEMPCTRL_FAN_DUTY_LOW))
				* TEMPCTRL_FAN_DUTY_MID;
		shared_cache.fan_on = true;
	}
	else if (diff < 5.0f)
	{
		// 0.5..5.0 => 0.4..1.0
		duty = TEMPCTRL_FAN_DUTY_MID + ((diff - 0.5f) / (5.0f - 0.5f)) * 0.6f;
		shared_cache.fan_on = true;
	}
	else
	{
		duty = 1.0f;
		shared_cache.fan_on = true;
	}

	duty = clampf(duty, 0.0f, 1.0f);
	FAN_SetPWM(duty, 2500);
	fan_duty = duty;
}

#include <stdint.h>
#include <stdbool.h>

// =====================
// Tunables
// =====================
// ---- PSU CV(전압) 입력 그래프 유효구간 ----
// 24~60V -> 1.0~4.7V (50%~125%)
#define PSU_VOUT_MIN_V           (24.0f)
#define PSU_VOUT_MAX_V           (60.0f)
#define PSU_CV_VMIN              (1.0f)
#define PSU_CV_VMAX              (4.7f)
#define PSU_CC_VMIN              (1.0f)
#define PSU_CC_VMAX              (4.7f)
#define PSU_VOUT_OFFSET				(-0.01f)
// ---- PSU CC(전류) 입력 그래프 유효구간 ----
// 20%~100% -> 1.0~4.7V (아래구간 비선형/plateau 가능)
#define PSU_I_MAX_A              (67.0f)   // <<< 여기: PSU 정격/실측 최대전류로 맞추는 게 핵심
#define PSU_I_MIN_FRAC           (0.20f)
#define PSU_I_MIN_A              (PSU_I_MAX_A * PSU_I_MIN_FRAC)

#define PSU_TASK_DT_SEC          (0.002f)     // task period

#define PSU_ZERO_A_DEADBAND_A    (0.05f)

// PV LPF (simple 1st-order IIR)
#define PSU_PV_ALPHA             (0.20f)

// DAC8845
#define DAC8845_VREF_V           (5.0f)

// ramp limiting for output pin voltage
#define PSU_RAMP_VPS             (2.0f)
#define PSU_PIN_EPS              (0.02f)

// CV compensation gain (Vpin per Vout error[V]) - you can tune
#define PSU_CV_KP_CORR           (3.7f/36.0f)
// 전류 PV 필터 (0=필터 없음, 0.1~0.3 권장)
#define PSU_I_PV_ALPHA           (0.05f)
#define I_ERR_DB_A  (0.015f) // 15mA 정도부터 반응

// =====================
// Helpers
// =====================
// CV: 24~60V -> 1.0~4.7V
// [목표 출력 전압(V), 그 출력을 내기 위해 필요한 실제 제어 전압(V)]
typedef struct
{
	double target_vout;
	double required_vctrl;
} psu_ctrl_calib_t;

// 반드시 target_vout 오름차순으로 정렬
static const psu_ctrl_calib_t psu_cv_table[] =
{
{ 24.0, 1.000 },
{ 40.0, 2.683 },
{ 41.0, 2.808 },
{ 42.0, 2.895 },
{ 43.0, 3.000 },
{ 45.0, 3.213 },
{ 60.0, 4.700 } };

#define PSU_CV_TBL_SIZE (sizeof(psu_cv_table)/sizeof(psu_cv_table[0]))

static inline double psu_cv_ctrl_volt_from_vout(double vout)
{
	vout = clampf(vout, PSU_VOUT_MIN_V, PSU_VOUT_MAX_V);

	// 테이블 기반 구간 보간
	for (int i = 0; i < PSU_CV_TBL_SIZE - 1; i++)
	{
		if (vout >= psu_cv_table[i].target_vout
				&& vout <= psu_cv_table[i + 1].target_vout)
		{
			double dx = psu_cv_table[i + 1].target_vout
					- psu_cv_table[i].target_vout;
			double dy = psu_cv_table[i + 1].required_vctrl
					- psu_cv_table[i].required_vctrl;

			double vctrl = psu_cv_table[i].required_vctrl
					+ (vout - psu_cv_table[i].target_vout) * (dy / dx);
			return clampf(vctrl, PSU_CV_VMIN, PSU_CV_VMAX) + PSU_VOUT_OFFSET;
		}
	}

	return PSU_CV_VMAX; // Fallback
}

// CC FF: (20%~100%) -> (1.0~4.7V)
static double psu_cc_ff_ctrl_volt_from_isv(double i_sv)
{
	i_sv = clampf(i_sv, 0.0f, PSU_I_MAX_A);

	// 0A는 별도 처리(보통 PSU_OFF가 정답이지만 여기선 DAC=0)
	if (i_sv <= PSU_ZERO_A_DEADBAND_A)
	{
		return 0.0f;
	}

	// 20% 미만은 plateau/비선형 가능 -> 최소 유효구간(1.0V)로 고정
	if (i_sv < PSU_I_MIN_A)
	{
		return PSU_CC_VMIN;
	}

	double f = i_sv / PSU_I_MAX_A; // 0~1
	// f:0.2~1.0 -> V:1.0~4.7
	double v = PSU_CC_VMIN
			+ (f - PSU_I_MIN_FRAC)
					* ((PSU_CC_VMAX - PSU_CC_VMIN) / (1.0f - PSU_I_MIN_FRAC));
	return clampf(v, PSU_CC_VMIN, PSU_CC_VMAX);
}
static double volt_from_4_20mA(double x_mA)
{
	// user formula: V = (X - 4.00) * 3.125
	double x = clampf(x_mA, 4.0f, 20.0f);
	return (x - 4.0f) * 3.125f; // 0..50V
}

static uint16_t dac8845_code_from_volt(double v)
{
	float vv = clampf(v, 0.0f, DAC8845_VREF_V);
	float code_f = (vv / DAC8845_VREF_V) * 65535.0f;
	if (code_f < 0.0f)
		code_f = 0.0f;
	if (code_f > 65535.0f)
		code_f = 65535.0f;
	return (uint16_t) (code_f + 0.5f);
}

static float ramp_toward(float cur, float target, float rate_vps, float dt)
{
	float max_step = rate_vps * dt;
	float diff = target - cur;
	if (diff > max_step)
		diff = max_step;
	if (diff < -max_step)
		diff = -max_step;
	return cur + diff;
}

static void psu_write_dacs(double vpin, double ipin)
{
	DAC_SetCode(1, dac8845_code_from_volt(vpin), 5); // CV pin
	DAC_SetCode(0, dac8845_code_from_volt(ipin), 5); // CC pin
}
// =====================
// NEW PI (fresh implementation)
// =====================
typedef struct
{
	float kp;        // Vpin per A error
	float ki;        // Vpin per (A*s)
	float integ;

	float out_min;   // correction clamp (V)
	float out_max;

	float kaw;       // anti-windup back-calc (0 disables)
	float rate_limit;       // V/s on correction output (0 disables)
	float y;         // last output (correction)
} pi_new_t;

static void pi_reset(pi_new_t *c, float y_init)
{
	c->integ = 0.0f;
	c->y = y_init;
}

static float pi_update(pi_new_t *c, float err, float dt)
{
	float up = c->kp * err;
	float u = up + c->integ;

	float u_sat = clampf(u, c->out_min, c->out_max);

	// anti-windup back-calc
	float integ_dot = (c->ki * err);
	if (c->kaw > 0.0f)
	{
		integ_dot += c->kaw * (u_sat - u);
	}
	c->integ += integ_dot * dt;

	// recompute + clamp
	u = up + c->integ;
	u_sat = clampf(u, c->out_min, c->out_max);

	// rate limit
	if (c->rate_limit > 0.0f)
	{
		float max_step = c->rate_limit * dt;
		float diff = u_sat - c->y;
		if (diff > max_step)
			diff = max_step;
		if (diff < -max_step)
			diff = -max_step;
		c->y += diff;
	}
	else
	{
		c->y = u_sat;
	}

	return c->y;
}

// =====================
// Local state
// =====================
static bool s_prev_psu_enable = false;
static bool s_ramping_down = false;

static double s_pin_v_out = 0.0f;   // last applied CV pin voltage [V]
static double s_pin_i_out = 0.0f;   // last applied CC pin voltage [V]

// current PV filter
static double s_i_pv_f = 0.0f;
static uint8_t s_i_init = 0;

// current loop PI (correction only)
static pi_new_t s_cc_pi =
{ .kp = 0.15f,        // <<< tune
		.ki = 0.80f,        // <<< tune
		.integ = 0.0f, .out_min = -0.8f,   // correction range (V)
		.out_max = +0.8f, .kaw = 2.0f,        // anti-windup strength
		.rate_limit = 3.0f, // correction slew (V/s), set 0 to disable
		.y = 0.0f };

// =====================
// Task E
// =====================
void Task_E_PSU(void)
{
	const double dt = PSU_TASK_DT_SEC;

	const bool emergency = shared_cache.is_emergency;
	const bool enabled = (!emergency && shared_cache.psu_enable);

	// 0) Emergency: immediate OFF
	if (emergency)
	{
		psu_write_dacs(0.0f, 0.0f);
		OUTCTL_Off(OUT_ELEC_CONTACT);
		OUTCTL_Off(OUT_PSU_ON);

		pi_reset(&s_cc_pi, 0.0f);

		s_prev_psu_enable = false;
		s_ramping_down = false;
		s_pin_v_out = 0.0f;
		s_pin_i_out = 0.0f;
		s_i_init = 0;
		return;
	}

	// 1) Disable (non-emergency): ramp-down then contactor OFF
	if (!enabled)
	{
		if (s_prev_psu_enable)
		{
			s_ramping_down = true;
			s_prev_psu_enable = false;
		}

		if (s_ramping_down)
		{
			s_pin_v_out = ramp_toward(s_pin_v_out, 0.0f, PSU_RAMP_VPS, dt);
			s_pin_i_out = ramp_toward(s_pin_i_out, 0.0f, PSU_RAMP_VPS, dt);
			psu_write_dacs(s_pin_v_out, s_pin_i_out);

			if (s_pin_v_out <= PSU_PIN_EPS && s_pin_i_out <= PSU_PIN_EPS)
			{
				OUTCTL_Off(OUT_ELEC_CONTACT);
				OUTCTL_Off(OUT_PSU_ON);

				pi_reset(&s_cc_pi, 0.0f);

				s_ramping_down = false;
				s_i_init = 0;
			}
		}
		else
		{
			psu_write_dacs(0.0f, 0.0f);
			OUTCTL_Off(OUT_ELEC_CONTACT);
			OUTCTL_Off(OUT_PSU_ON);
		}
		return;
	}

	// 2) Rising edge init
	if (!s_prev_psu_enable && enabled)
	{
		pi_reset(&s_cc_pi, 0.0f);
		s_ramping_down = false;
		s_i_init = 0;
	}
	s_prev_psu_enable = true;

	OUTCTL_On(OUT_ELEC_CONTACT);
	OUTCTL_On(OUT_PSU_ON);

	// 3) PV read
	(void) volt_from_4_20mA(shared_cache.volt_sen_mA); // monitor only (필요하면 로깅)
	double i_pv = shared_cache.curr_sen_A;

	// 4) current PV filter
	if (!s_i_init)
	{
		s_i_pv_f = i_pv;
		s_i_init = 1;
	}
	s_i_pv_f += PSU_I_PV_ALPHA * (i_pv - s_i_pv_f);

	// 5) SV clamp
	double v_sv = clampf(shared_cache.voltage_sv, PSU_VOUT_MIN_V,
	PSU_VOUT_MAX_V);
	double i_sv = clampf(shared_cache.current_sv, 0.0f, PSU_I_MAX_A);

	// ------------------------------------------------------------
	// A) Voltage OPEN-LOOP (FF only)
	// ------------------------------------------------------------
	double vpin = psu_cv_ctrl_volt_from_vout(v_sv);
	vpin = clampf(vpin, PSU_CV_VMIN, PSU_CV_VMAX);

	// ------------------------------------------------------------
	// B) Current CLOSED-LOOP (FF + PI correction)
	// ------------------------------------------------------------
	double ipin;

	if (i_sv <= PSU_ZERO_A_DEADBAND_A)
	{
		// 0A: CC pin 0V, reset PI
		pi_reset(&s_cc_pi, 0.0f);
		ipin = 0.0f;

		// mode flags: effectively CV only
		shared_cache.psu48_mode_cv_ctrl = true;
		shared_cache.psu48_mode_cc_ctrl = false;
	}
	else
	{
		double ipin_ff = psu_cc_ff_ctrl_volt_from_isv(i_sv);

		double ierr = i_sv - s_i_pv_f;
		if (fabsf(ierr) < I_ERR_DB_A)
		{
			ierr = 0.0f;
		}
		double ipin_corr = pi_update(&s_cc_pi, ierr, dt);

		ipin = ipin_ff + ipin_corr;
		ipin = clampf(ipin, PSU_CC_VMIN, PSU_CC_VMAX);

		// mode flags: CC loop active
		shared_cache.psu48_mode_cv_ctrl = false;
		shared_cache.psu48_mode_cc_ctrl = true;
	}

	// 6) Apply with ramp limiting (prevents steps)
	s_pin_v_out = ramp_toward(s_pin_v_out, vpin, PSU_RAMP_VPS, dt);
	s_pin_i_out = ramp_toward(s_pin_i_out, ipin, PSU_RAMP_VPS, dt);

	psu_write_dacs(s_pin_v_out, s_pin_i_out);
}

// 현재 모드에 맞게 밸브 상태를 실제로 적용
void Task_F_ApplyValveMode()
{
	if (shared_cache.is_emergency)
	{
		OUTCTL_On(OUT_H_GAS_SOL1);
		OUTCTL_On(OUT_H_GAS_SOL2);
		OUTCTL_On(OUT_H_GAS_SOL3);
		shared_cache.highp_production_ctrl = false;
		return;
	}
	switch (shared_cache.valve_mode)
	{
	case VALVE_MODE_ATMOS:
		// 상압 모드: V1 open, V2 close, V3 close
		OUTCTL_On(OUT_H_GAS_SOL1);
		OUTCTL_Off(OUT_H_GAS_SOL2);
		OUTCTL_Off(OUT_H_GAS_SOL3);
		shared_cache.highp_production_ctrl = false;
		break;

	case VALVE_MODE_HIGHP:
		// 고압 모드: V1 close, V2 open, V3 close
		OUTCTL_Off(OUT_H_GAS_SOL1);
		OUTCTL_On(OUT_H_GAS_SOL2);
		OUTCTL_Off(OUT_H_GAS_SOL3);
		shared_cache.highp_production_ctrl = true;
		break;
	default:
		break;
	}
}
void Task_L_UpdateLEDs(void)
{
	static uint8_t blinking = 0;
	static uint32_t last_blink_time = 0;
	if (shared_cache.is_emergency)
	{
		if (shared_cache.time_sync_ok == false)
		{
			if (HAL_GetTick() - last_blink_time > 250)
			{
				blinking = !blinking;
				last_blink_time = HAL_GetTick();
			}
			if (blinking)
			{
				ws2812SetAllColor(WS2812_COLOR_EMERGENCY);
			}
			else
			{
				ws2812SetAllColor(0x000000);
			}
		}
		else
		{
			ws2812SetAllColor(WS2812_COLOR_EMERGENCY);
		}
	}
	else if (shared_cache.electrolyte_enable && shared_cache.psu_enable)
	{
		ws2812SetAllColor(WS2812_COLOR_MAKE_H2);
	}
	else if (shared_cache.electrolyte_enable)
	{
		ws2812SetAllColor(WS2812_COLOR_ELECTROLYTE_CONTROL_ON);
	}
	else if (shared_cache.psu_enable)
	{
		ws2812SetAllColor(WS2812_COLOR_POEWR_CONTROL_ON);
	}
	else
	{
		ws2812SetAllColor(WS2812_COLOR_STANDBY);
	}
}

// [Task Supervisor] 전체 상태 관리
void Task_Supervisor(void)
{
	volatile uint32_t now = HAL_GetTick();

	// 1. 통신 타임아웃 체크 (3초)
	if (now - g_hxb2_cmd_state.tick_lastbytereceived > 15000
			&& HAL_GPIO_ReadPin(GPIOA, GPIO_PIN_11) == GPIO_PIN_SET)
	{
		shared_cache.time_sync_ok = false;
		shared_cache.is_emergency = true; // 통신 두절 시 비상 정지
	}
	else
	{
		shared_cache.time_sync_ok = true;
	}

	// 2. 비상 정지 시 안전 조치
	if (shared_cache.is_emergency)
	{
		shared_cache.psu_enable = false;
		shared_cache.electrolyte_enable = false;

		// 즉시 하드웨어 차단
		//mcp4728_Write_All_Channels(0, 0, 0, 0);
		//MCP4728_Write_AllChannels_Same(&hi2c3, 0);
		//All_Valves_Off();
	}
	if (HAL_GPIO_ReadPin(SWITCH_GPIO_Port, SWITCH_Pin) == GPIO_PIN_RESET)
	{
		NVIC_SystemReset();
	}
}

// =============================================================
// 3. 메인 로직 루프 (main.c에서 호출)
// =============================================================
/*
 void System_Logic_Loop(void)
 {
 uint32_t now = HAL_GetTick();

 if (DIN_SWITCH() == false)
 {
 HAL_NVIC_SystemReset();
 }
 // 항상 실행 (데이터 수집 & 감시)
 if (now - tick_A >= PERIOD_TASK_A_SENSORS)
 {
 Task_A_Sensors();
 tick_A = now;
 }

 if (now - tick_Sup >= PERIOD_SUPERVISOR)
 {
 Task_Supervisor();
 tick_Sup = now;
 }

 // 제어 실행 (비상 정지가 아닐 때만)

 {

 if (now - tick_B >= PERIOD_TASK_B_REFILL)
 {
 Task_B_Refill();
 tick_B = now;
 }

 if (now - tick_C >= PERIOD_TASK_C_CIRC)
 {
 Task_C_Circulation();
 tick_C = now;
 }

 if (now - tick_D >= PERIOD_TASK_D_TEMP)
 {
 Task_D_TempControl();
 tick_D = now;
 }

 if (now - tick_E >= PERIOD_TASK_E_PSU)
 {
 Task_E_PSU();
 tick_E = now;
 }
 if (now - tick_F >= PERIOD_TASK_F)
 {
 Task_F_ApplyValveMode();
 tick_F = now;
 }
 }
 }*/
