#ifndef MODBUS_H
#define MODBUS_H

#include <stdint.h>
#include <stdbool.h>
#include "main.h"

#define MODBUS_DEBUG
#define DEBUG_MODE

// 레지스터 크기 매크로 정의
#define HOLDING_REGISTERS_SIZE   100
#define INPUT_REGISTERS_SIZE     100
#define COILS_SIZE               100
#define DISCRETE_INPUTS_SIZE     100

// 버퍼 크기 정의
#define BUFFER_SIZE 256
#define NUM_RING_BUFFERS 2

// 모드버스 함수 코드
#define MODBUS_FC_READ_COILS               0x01
#define MODBUS_FC_READ_DISCRETE_INPUTS     0x02
#define MODBUS_FC_READ_HOLDING_REGISTERS   0x03
#define MODBUS_FC_READ_INPUT_REGISTERS     0x04
#define MODBUS_FC_WRITE_SINGLE_COIL        0x05
#define MODBUS_FC_WRITE_SINGLE_REGISTER    0x06
#define MODBUS_FC_WRITE_MULTIPLE_COILS     0x0F
#define MODBUS_FC_WRITE_MULTIPLE_REGISTERS 0x10

// 모드버스 예외 코드
#define MODBUS_EXCEPTION_ILLEGAL_FUNCTION      0x01
#define MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS  0x02
#define MODBUS_EXCEPTION_ILLEGAL_DATA_VALUE    0x03
#define MODBUS_EXCEPTION_SLAVE_DEVICE_FAILURE  0x04

#ifdef MODBUS_DEBUG
/* 디버깅용 홀딩 레지스터 주소 매크로 정의 */
#define DEBUG_RX_STATUS_ADDR        90  // 수신 상태 코드
#define DEBUG_TX_STATUS_ADDR        91  // 송신 상태 코드
#define DEBUG_RX_LENGTH_ADDR        92  // 수신 데이터 길이
#define DEBUG_BUFFER_STATUS_ADDR    93  // 버퍼 상태
#define DEBUG_PARSE_STATUS_ADDR     94  // 파싱 상태
#define DEBUG_ERROR_COUNT_ADDR      95  // 에러 카운터
#define DEBUG_LAST_ERROR_CODE_ADDR  96  // 마지막 에러 코드
#define DEBUG_RX_COUNT_ADDR         97  // 수신 프레임 카운터
#define DEBUG_TX_LENGTH_ADDR        98  // 송신 데이터 길이
#define DEBUG_FRAME_LENGTH_ADDR     99  // 현재 프레임 길이

#endif
// 모드버스 모드 열거형
typedef enum {
    MODBUS_MODE_RTU = 0,
    MODBUS_MODE_ASCII = 1,
    MODBUS_MODE_TCP = 2,
    MODBUS_MODE_ETC = 3
} ModbusMode;

// 모드버스 상태 열거형
typedef enum {
    // 버퍼 상태
    RINGBUF_OK = 0,
    RINGBUF_OVERRUN = 1,
    
    // 수신 상태
    RX_IDLE = 10,
    RX_IN_PROGRESS = 11,
    RX_COMPLETE = 12,
    RX_ERROR_LENGTH = 13,
    RX_ERROR_START = 14,
    RX_ERROR_CRC = 15,
    RX_ERROR = 16,
    RX_ERROR_OVERFLOW = 17,
    
    // 송신 상태
    TX_IDLE = 20,
    TX_IN_PROGRESS = 21,
    TX_COMPLETE = 22,
    TX_ERROR = 23,
    
    // 파싱 상태
    PARSE_OK = 30,
    PARSE_ERROR_FUNC = 31,
    PARSE_ERROR_ADDR = 32,
    PARSE_ERROR_VALUE = 33,
    PARSE_ERROR_LENGTH = 34,
    PARSE_ERROR_LENGTH2 = 35,
    RX_RECOVERY_TRIGGERED = 36

} ModbusStatus;

// 프레임 구조체
typedef struct {
    uint16_t length;
    uint8_t data[BUFFER_SIZE];
} Frame_t;

// 레지스터 구조체
typedef struct {
    uint32_t holdingRegisters[HOLDING_REGISTERS_SIZE];
    uint32_t inputRegisters[INPUT_REGISTERS_SIZE];
    uint8_t coils[COILS_SIZE];
    uint8_t discreteInputs[DISCRETE_INPUTS_SIZE];
} ModbusRegisters_t;

// 버퍼 관리 구조체
typedef struct {
    Frame_t rxFrameBuffer;
    Frame_t currentFrame;
    Frame_t txFrameBuffer;
    Frame_t rxRingFrameBuffer[NUM_RING_BUFFERS];
    uint8_t rxWriteIndex;
    uint8_t rxReadIndex;
    bool frameReadyFlag;
    uint8_t firstDataFlag;
} ModbusBuffers_t;

// 상태 관리 구조체
typedef struct {
    ModbusStatus rxStatus;
    ModbusStatus txStatus;
    ModbusStatus parseStatus;
    ModbusStatus bufferStatus;
    uint16_t errorCounter;
    ModbusStatus lastErrorCode;
    ModbusMode currentRxMode;
    ModbusMode currentParseMode;
} ModbusStatus_t;

#ifdef ASCII_MODE
// ASCII 모드 관련 구조체
typedef struct {
    uint16_t timeoutCounter;
    bool receptionActive;
} ModbusAscii_t;


// 모드버스 전체 구조체
typedef struct {
    uint8_t slaveId;
    UART_HandleTypeDef *huart;
    ModbusRegisters_t registers;
    ModbusBuffers_t buffers;
    ModbusStatus_t status;
    ModbusAscii_t *ascii;
} Modbus_t;
#else
// 모드버스 전체 구조체
typedef struct {
    uint8_t slaveId;
    UART_HandleTypeDef *huart;
    ModbusRegisters_t registers;
    ModbusBuffers_t buffers;
    ModbusStatus_t status;
} Modbus_t;
#endif

#ifdef ASCII_MODE
// 초기화 함수
void Modbus_Init(Modbus_t *modbus, UART_HandleTypeDef *huart, uint8_t slaveId, ModbusAscii_t *ascii);
#else
void Modbus_Init(Modbus_t *modbus, UART_HandleTypeDef *huart, uint8_t slaveId);
#endif

// 메인 처리 함수
void Modbus_Process(Modbus_t *modbus);

// 인터럽트 콜백 함수
void Modbus_UARTRxCallback(Modbus_t *modbus, UART_HandleTypeDef *huart);
void Modbus_UARTTxCallback(Modbus_t *modbus, UART_HandleTypeDef *huart);
void Modbus_UARTIdleCallback(Modbus_t *modbus, UART_HandleTypeDef *huart);

// 레지스터 접근 함수
uint16_t Modbus_GetHoldingRegister(Modbus_t *modbus, uint16_t address);
void Modbus_SetHoldingRegister(Modbus_t *modbus, uint16_t address, uint16_t value);
uint16_t Modbus_GetInputRegister(Modbus_t *modbus, uint16_t address);
void Modbus_SetInputRegister(Modbus_t *modbus, uint16_t address, uint16_t value);
uint8_t Modbus_GetCoil(Modbus_t *modbus, uint16_t address);
void Modbus_SetCoil(Modbus_t *modbus, uint16_t address, uint8_t value);
uint8_t Modbus_GetDiscreteInput(Modbus_t *modbus, uint16_t address);
void Modbus_SetDiscreteInput(Modbus_t *modbus, uint16_t address, uint8_t value);

// 유틸리티 함수
void Modbus_CreateExceptionResponse(Modbus_t *modbus, uint8_t functionCode, uint8_t exceptionCode);
void Modbus_PrepareResponseData(Modbus_t *modbus);
void Modbus_SendResponseData(Modbus_t *modbus);

bool push_frame(Modbus_t *modbus, const Frame_t *newFrame);
bool pop_frame(Modbus_t *modbus, Frame_t *outFrame);

void Modbus_SetSlaveId(Modbus_t *modbus, uint8_t slaveId);
void Modbus_ResetRx(Modbus_t *modbus);
#endif
