/*
 * dac8554.h
 *
 *  Created on: Feb 19, 2026
 *      Author: c6h6
 */

#ifndef DAC8554_H_
#define DAC8554_H_

#pragma once

#include "main.h"
#include <stdint.h>
#include <stdbool.h>


/*
 * ===== User Fixed HW Binding =====
 * - SPI instance  : SPI2  (HAL handle: hspi2)
 * - SYNC(/CS) pin : PB14
 * - LDAC pin      : PB12
 *
 * NOTE:
 *  - CS(SYNC)는 active-low.
 *  - LDAC은 Low->High rising edge에 업데이트 트리거.
 */
#ifdef __cplusplus
extern "C" {
#endif

#ifdef TASKS_AS_FREERTOS
#include "FreeRTOS.h"
#include "task.h"
typedef enum {
    DAC_SVC_REQ_WRITE_UPDATE = 0,
    DAC_SVC_REQ_WRITE_BUFFER,
    DAC_SVC_REQ_TRIGGER_LDAC,
    DAC_SVC_REQ_POWERDOWN,
} dac_svc_req_type_t;

typedef enum {
    DAC_SVC_PD_HIZ_0    = 0,
    DAC_SVC_PD_1K_GND   = 1,
    DAC_SVC_PD_100K_GND = 2,
    DAC_SVC_PD_HIZ_1    = 3,
} dac_svc_pdmode_t;

typedef struct {
    dac_svc_req_type_t type;
    uint8_t            ch;        // 0..3
    uint16_t           code16;    // 0..65535
    dac_svc_pdmode_t   pd_mode;
    bool               load_now;
    TaskHandle_t       caller;    // 내부 사용
} dac_svc_req_t;


HAL_StatusTypeDef DAC_SetCode(uint8_t ch, uint16_t code16, uint32_t timeout_ms);
HAL_StatusTypeDef DAC_SetCodeBuffered(uint8_t ch, uint16_t code16, uint32_t timeout_ms);
HAL_StatusTypeDef DAC_Commit(uint32_t timeout_ms);
HAL_StatusTypeDef DAC_SetCodesSimul(uint16_t a, uint16_t b, uint16_t c, uint16_t d, uint32_t timeout_ms);
HAL_StatusTypeDef DAC_PowerDown(uint8_t ch, dac_svc_pdmode_t mode, bool load_now, uint32_t timeout_ms);
// 큐 길이(필요시 조정)
#ifndef DAC_SVC_QUEUE_LEN
#define DAC_SVC_QUEUE_LEN  16
#endif
#endif

// CubeMX가 만들어주는 SPI2 핸들
extern SPI_HandleTypeDef hspi2;

// ---- Pin bindings ----
#define DAC8554_SPI              (&hspi2)

#define DAC8554_SYNC_PORT        GPIOB
#define DAC8554_SYNC_PIN         GPIO_PIN_14

#define DAC8554_LDAC_PORT        GPIOB
#define DAC8554_LDAC_PIN         GPIO_PIN_12

// A1/A0 하드웨어 주소핀 (보통 GND=0)
#ifndef DAC8554_ADDR_A1
#define DAC8554_ADDR_A1          (0u)
#endif
#ifndef DAC8554_ADDR_A0
#define DAC8554_ADDR_A0          (0u)
#endif

// LDAC 핀을 실제로 MCU가 제어하는지 (GND에 묶였으면 0으로)
#ifndef DAC8554_USE_LDAC_PIN
#define DAC8554_USE_LDAC_PIN     (1u)
#endif

// SPI transmit timeout (ms)
#ifndef DAC8554_SPI_TIMEOUT_MS
#define DAC8554_SPI_TIMEOUT_MS   (10u)
#endif

typedef enum {
    DAC8554_CH_A = 0,
    DAC8554_CH_B = 1,
    DAC8554_CH_C = 2,
    DAC8554_CH_D = 3,
} dac8554_ch_t;

typedef enum {
    DAC8554_PD_HIZ_0    = 0, // 100 or 111 -> Hi-Z
    DAC8554_PD_1K_GND   = 1, // 101 -> ~1k to GND
    DAC8554_PD_100K_GND = 2, // 110 -> ~100k to GND
    DAC8554_PD_HIZ_1    = 3, // 111 -> Hi-Z
} dac8554_pdmode_t;

// 초기화(핀 idle 상태 세팅)
void DAC8554_Init(void);

// 즉시 업데이트 (LD1=0, LD0=1)
HAL_StatusTypeDef DAC8554_WriteUpdate(dac8554_ch_t ch, uint16_t code);

// 버퍼만 로드 (LD1=0, LD0=0)
HAL_StatusTypeDef DAC8554_WriteBuffer(dac8554_ch_t ch, uint16_t code);

// 소프트웨어 동시 업데이트 (LD1=1, LD0=0)
HAL_StatusTypeDef DAC8554_SoftwareSimulUpdate(dac8554_ch_t ch, uint16_t code);

// 파워다운 설정
HAL_StatusTypeDef DAC8554_PowerDown(dac8554_ch_t ch, dac8554_pdmode_t mode, bool load_now);

// LDAC rising edge 트리거(하드웨어 동시 업데이트)
HAL_StatusTypeDef DAC8554_TriggerLDAC(void);




#ifdef __cplusplus
}
#endif


#endif /* DAC8554_H_ */
