/*
 * dac8554.c
 *
 *  Created on: Feb 19, 2026
 *      Author: c6h6
 */


#include "dac8554.h"
#include "task.h"
#include "queue.h"
QueueHandle_t      s_q    = NULL;
TaskHandle_t       s_task = NULL;

// --- GPIO helpers ---
static inline void sync_low(void)  { HAL_GPIO_WritePin(DAC8554_SYNC_PORT, DAC8554_SYNC_PIN, GPIO_PIN_RESET); }
static inline void sync_high(void) { HAL_GPIO_WritePin(DAC8554_SYNC_PORT, DAC8554_SYNC_PIN, GPIO_PIN_SET);   }

static inline void ldac_low(void)  { HAL_GPIO_WritePin(DAC8554_LDAC_PORT, DAC8554_LDAC_PIN, GPIO_PIN_RESET); }
static inline void ldac_high(void) { HAL_GPIO_WritePin(DAC8554_LDAC_PORT, DAC8554_LDAC_PIN, GPIO_PIN_SET);   }

static uint8_t ch_sel_bits(dac8554_ch_t ch)
{
    return ((uint8_t)ch) & 0x03u; // A:0 B:1 C:2 D:3
}
dac8554_ch_t map_ch(uint8_t ch)
{
    return ch_sel_bits(ch);
}

static void build_frame(uint8_t ld1, uint8_t ld0,
                        dac8554_ch_t ch,
                        uint8_t pd0, uint8_t pd2, uint8_t pd1,
                        uint16_t data16,
                        uint8_t out[3])
{
    // DB23..DB16 = [A1 A0 LD1 LD0 X Sel1 Sel0 PD0]
    uint8_t sel = ch_sel_bits(ch);

    uint8_t db23_16 =
        ((DAC8554_ADDR_A1 & 1u) << 7) |
        ((DAC8554_ADDR_A0 & 1u) << 6) |
        ((ld1 & 1u)             << 5) |
        ((ld0 & 1u)             << 4) |
        (0u                     << 3) |
        (((sel >> 1) & 1u)      << 2) |
        ((sel & 1u)             << 1) |
        ((pd0 & 1u)             << 0);

    uint16_t db15_0 = data16;

    if (pd0) {
        // PD0=1이면 DB15..DB14가 PD2..PD1 (나머지는 don't care)
        db15_0 = (uint16_t)(((pd2 & 1u) << 15) | ((pd1 & 1u) << 14));
    }

    // MSB first 24-bit
    out[0] = db23_16;
    out[1] = (uint8_t)((db15_0 >> 8) & 0xFFu);
    out[2] = (uint8_t)(db15_0 & 0xFFu);
}

static HAL_StatusTypeDef spi_write24(const uint8_t buf[3])
{
    // SYNC low 동안 24bit shift 후 SYNC high
    sync_low();
    HAL_StatusTypeDef st = HAL_SPI_Transmit(DAC8554_SPI, (uint8_t*)buf, 3, DAC8554_SPI_TIMEOUT_MS);
    sync_high();
    return st;
}

void DAC8554_Init(void)
{
    // idle 상태
    sync_high();

#if (DAC8554_USE_LDAC_PIN)
    // 보통 LDAC은 LOW에 두고, 필요할 때 HIGH 펄스로 트리거
    ldac_low();
#endif
}

HAL_StatusTypeDef DAC8554_WriteUpdate(dac8554_ch_t ch, uint16_t code)
{
    uint8_t f[3];
    // LD1=0 LD0=1 : single-channel update
    build_frame(0, 1, ch, 0, 0, 0, code, f);
    return spi_write24(f);
}

HAL_StatusTypeDef DAC8554_WriteBuffer(dac8554_ch_t ch, uint16_t code)
{
    uint8_t f[3];
    // LD1=0 LD0=0 : write buffer only
    build_frame(0, 0, ch, 0, 0, 0, code, f);
    return spi_write24(f);
}

HAL_StatusTypeDef DAC8554_SoftwareSimulUpdate(dac8554_ch_t ch, uint16_t code)
{
    uint8_t f[3];
    // LD1=1 LD0=0 : software simultaneous update
    build_frame(1, 0, ch, 0, 0, 0, code, f);
    return spi_write24(f);
}

HAL_StatusTypeDef DAC8554_PowerDown(dac8554_ch_t ch, dac8554_pdmode_t mode, bool load_now)
{
    uint8_t pd2 = (uint8_t)((mode >> 1) & 1u);
    uint8_t pd1 = (uint8_t)(mode & 1u);

    uint8_t ld1 = 0;
    uint8_t ld0 = load_now ? 1u : 0u;

    uint8_t f[3];
    build_frame(ld1, ld0, ch, 1u, pd2, pd1, 0u, f);
    return spi_write24(f);
}

HAL_StatusTypeDef DAC8554_TriggerLDAC(void)
{
#if !(DAC8554_USE_LDAC_PIN)
    return HAL_ERROR;
#else
    // Low->High rising edge가 트리거이므로, HIGH 펄스 후 LOW 복귀
    ldac_high();
    for (volatile int i = 0; i < 50; i++) { __NOP(); } // ~몇 us 정도(클럭에 따라)
    ldac_low();
    return HAL_OK;
#endif
}

#ifdef TASKS_AS_FREERTOS
static inline TickType_t to_ticks(uint32_t timeout_ms)
{
    return (timeout_ms == 0) ? 0 : pdMS_TO_TICKS(timeout_ms);
}

void DAC_TaskInit(void)
{
    if (s_q == NULL) {
        s_q = xQueueCreate(DAC_SVC_QUEUE_LEN, sizeof(dac_svc_req_t));
        // 실패 시 NULL일 수 있으니, 필요하면 assert 처리
    }
}

static HAL_StatusTypeDef post_and_wait(dac_svc_req_t *r, uint32_t timeout_ms)
{
    if (s_q == NULL) return HAL_ERROR;

    TickType_t to = to_ticks(timeout_ms);
    r->caller = xTaskGetCurrentTaskHandle();

    if (xQueueSend(s_q, r, to) != pdTRUE) {
        return HAL_TIMEOUT;
    }

    uint32_t note = 0;
    if (xTaskNotifyWait(0, 0xFFFFFFFFu, &note, to) != pdTRUE) {
        return HAL_TIMEOUT;
    }

    return (HAL_StatusTypeDef)note;
}

// ===== Public Sync APIs =====

HAL_StatusTypeDef DAC_SetCode(uint8_t ch, uint16_t code16, uint32_t timeout_ms)
{
    dac_svc_req_t r = {
        .type = DAC_SVC_REQ_WRITE_UPDATE,
        .ch = (uint8_t)(ch & 0x03u),
        .code16 = code16,
        .pd_mode = DAC_SVC_PD_HIZ_0,
        .load_now = false,
        .caller = NULL
    };
    return post_and_wait(&r, timeout_ms);
}

HAL_StatusTypeDef DAC_SetCodeBuffered(uint8_t ch, uint16_t code16, uint32_t timeout_ms)
{
    dac_svc_req_t r = {
        .type = DAC_SVC_REQ_WRITE_BUFFER,
        .ch = (uint8_t)(ch & 0x03u),
        .code16 = code16,
        .pd_mode = DAC_SVC_PD_HIZ_0,
        .load_now = false,
        .caller = NULL
    };
    return post_and_wait(&r, timeout_ms);
}

HAL_StatusTypeDef DAC_Commit(uint32_t timeout_ms)
{
    dac_svc_req_t r = {
        .type = DAC_SVC_REQ_TRIGGER_LDAC,
        .ch = 0,
        .code16 = 0,
        .pd_mode = DAC_SVC_PD_HIZ_0,
        .load_now = false,
        .caller = NULL
    };
    return post_and_wait(&r, timeout_ms);
}

HAL_StatusTypeDef DAC_SetCodesSimul(uint16_t a, uint16_t b, uint16_t c, uint16_t d, uint32_t timeout_ms)
{
    // 4채널 버퍼 로드 후 LDAC 트리거를 “순차 동기 호출”로 묶음
    // (중간에 다른 태스크 명령이 끼어들면 동시성이 깨지므로,
    //  아래는 "하나의 요청"으로 만들면 더 완벽하지만,
    //  간단 버전으로는 timeout을 짧게, 그리고 호출 빈도를 낮추는 걸 권장)
    //
    // 더 완벽하게 하려면: DAC_SVC_REQ_SET4_SIMUL 같은 요청 타입을 추가해서
    // Task 내부에서 5번 동작을 한 번에 처리하게 만들면 됨.

    HAL_StatusTypeDef st;
    st = DAC_SetCodeBuffered(0, a, timeout_ms); if (st != HAL_OK) return st;
    st = DAC_SetCodeBuffered(1, b, timeout_ms); if (st != HAL_OK) return st;
    st = DAC_SetCodeBuffered(2, c, timeout_ms); if (st != HAL_OK) return st;
    st = DAC_SetCodeBuffered(3, d, timeout_ms); if (st != HAL_OK) return st;
    st = DAC_Commit(timeout_ms);                if (st != HAL_OK) return st;

    return HAL_OK;
}

HAL_StatusTypeDef DAC_PowerDown(uint8_t ch, dac_svc_pdmode_t mode, bool load_now, uint32_t timeout_ms)
{
    dac_svc_req_t r = {
        .type = DAC_SVC_REQ_POWERDOWN,
        .ch = (uint8_t)(ch & 0x03u),
        .code16 = 0,
        .pd_mode = mode,
        .load_now = load_now,
        .caller = NULL
    };
    return post_and_wait(&r, timeout_ms);
}
#endif

